<?php
/**
 * Dynamic Ads + Pixel
 *
 * @author    businesstech.fr <modules@businesstech.fr> - https://www.businesstech.fr/
 * @copyright Business Tech - https://www.businesstech.fr/
 * @license   see file: LICENSE.txt
 *
 *           ____    _______
 *          |  _ \  |__   __|
 *          | |_) |    | |
 *          |  _ <     | |
 *          | |_) |    | |
 *          |____/     |_|
 */

namespace FacebookProductAd\ModuleLib;

use FacebookProductAd\Dao\moduleDao;
use FacebookProductAd\Pixel\basePixel;
use FacebookProductAd\Configuration\moduleConfiguration;
use FacebookProductAd\Models\Reporting;
use FacebookProductAd\Models\Feeds;
use FacebookProductAd\ModuleLib\moduleUpdate;

class moduleTools
{
    /**
     * Magic Method __construct
     */
    private function __construct()
    {
    }

    /**
     * method returns good translated errors
     */
    public static function translateJsMsg()
    {
        return moduleConfiguration::getJsMessage();
    }

    /**
     * method update new keys in new module version
     */
    public static function updateConfiguration()
    {
        // check to update new module version
        foreach (moduleConfiguration::getConfVar() as $sKey => $mVal) {
            // use case - not exists
            if (\Configuration::get($sKey) === false) {
                // update key/ value
                \Configuration::updateValue($sKey, $mVal);
            }
        }
    }

    /**
     * method set all constant module in ps_configuration
     *
     * @param array $aOptionListToUnserialize
     * @param int $iShopId
     */
    public static function getConfiguration(array $aOptionListToUnserialize = null, $iShopId = null)
    {
        // get configuration options
        if (null !== $iShopId && is_numeric($iShopId)) {
            \FacebookProductAd::$conf = \Configuration::getMultiple(array_keys(moduleConfiguration::getConfVar()), null, null, $iShopId);
        } else {
            \FacebookProductAd::$conf = \Configuration::getMultiple(array_keys(moduleConfiguration::getConfVar()));
        }
        if (
            !empty($aOptionListToUnserialize)
            && is_array($aOptionListToUnserialize)
        ) {
            foreach ($aOptionListToUnserialize as $sOption) {
                if (!empty(\FacebookProductAd::$conf[strtoupper($sOption)]) && is_string(\FacebookProductAd::$conf[strtoupper($sOption)]) && !is_numeric(\FacebookProductAd::$conf[strtoupper($sOption)])) {
                    \FacebookProductAd::$conf[strtoupper($sOption)] = unserialize(\FacebookProductAd::$conf[strtoupper($sOption)]);
                }
            }
        }
    }

    /**
     * method set good iso lang
     *
     * @return string
     */
    public static function getLangIso($iLangId = null)
    {
        if (null === $iLangId) {
            $iLangId = \FacebookProductAd::$iCurrentLang;
        }

        // get iso lang
        $sIsoLang = \Language::getIsoById($iLangId);

        if (false === $sIsoLang) {
            $sIsoLang = 'en';
        }
        return $sIsoLang;
    }

    /**
     * method return Lang id from iso code
     *
     * @param string $sIsoCode
     * @return int
     */
    public static function getLangId($sIsoCode, $iDefaultId = null)
    {
        // get iso lang
        $iLangId = \Language::getIdByIso($sIsoCode);

        if (empty($iLangId) && $iDefaultId !== null) {
            $iLangId = $iDefaultId;
        }
        return $iLangId;
    }


    /**
     * method Performs a "UNION" of installed shop languages and languages
     * default supported values
     *
     * @param int $iShopId
     * @return array
     */
    public static function getAvailableLanguages($iShopId)
    {
        // set
        $aAvailableLanguages = array();

        $aShopLanguages = \Language::getLanguages(false, (int) ($iShopId));

        foreach ($aShopLanguages as $aLanguage) {
            if ($aLanguage['active']) {
                $aAvailableLanguages[] = $aLanguage;
            }
        }

        return $aAvailableLanguages;
    }

    /**
     * returns information about languages / countries and currencies available for Google
     *
     * @param array $available_languages
     * @return array
     */
    public static function getLangCurrencyCountry(array $available_languages)
    {
        // Force database update to be sure we could make the migration
        moduleUpdate::create()->run('tables');
        $output_data = array();

        $hasData = Feeds::hasSavedData(\FacebookProductAd::$iShopId);
        if (!empty($hasData)) {

            $available_feeds = Feeds::getAvailableFeeds((int)\FacebookProductAd::$iShopId);
            if (!empty($available_feeds)) {
                foreach ($available_languages as $lang) {

                    $current_feed_shop = Feeds::getFeedLangData($lang['iso_code'], (int)\FacebookProductAd::$iShopId);

                    if (isset($current_feed_shop)) {
                        foreach ($current_feed_shop as $feed) {

                            $language = new \Language($lang['id_lang']);
                            $id_country = \Country::getByIso(\Tools::strtolower($feed['iso_country']));

                            if (!empty($id_country)) {
                                $country_name = \Country::getNameById(\FacebookProductAd::$iCurrentLang, $id_country);
                                $country = new \Country($id_country);

                                if (!empty($country->id)) {
                                    if (!empty($country->active)) {
                                        $id_currency = \Currency::getIdByIsoCode($feed['iso_currency']);
                                        $currency = new \Currency($id_currency);
                                        if (!empty($currency->iso_code)) {
                                            $output_data[] = array(
                                                'langId' => $language->id,
                                                'langIso' => $language->iso_code,
                                                'countryIso' => $country->iso_code,
                                                'currencyIso' => $currency->iso_code,
                                                'currencyId' => $currency->id,
                                                'currencyFirst' => 1,
                                                'langName' => $language->name,
                                                'countryName' => $country_name,
                                                'currencySign' => $currency->sign,
                                                'taxonomy' => $feed['taxonomy'],
                                            );
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        sort($output_data);
        return $output_data;
    }


    /**
     * method returns current currency sign or id
     *
     * @param string $sField : field name has to be returned
     * @param string $iCurrencyId : currency id
     * @return mixed : string or array
     */
    public static function getCurrency($sField = null, $iCurrencyId = null)
    {
        // set
        $mCurrency = null;

        // get currency id
        if (null === $iCurrencyId) {
            $iCurrencyId = \Configuration::get('PS_CURRENCY_DEFAULT');
        }

        $aCurrency = \Currency::getCurrency($iCurrencyId);

        if ($sField !== null) {
            switch ($sField) {
                case 'id_currency':
                    $mCurrency = $aCurrency['id_currency'];
                    break;
                case 'name':
                    $mCurrency = $aCurrency['name'];
                    break;
                case 'iso_code':
                    $mCurrency = $aCurrency['iso_code'];
                    break;
                case 'iso_code_num':
                    $mCurrency = $aCurrency['iso_code_num'];
                    break;
                case 'sign':
                    $mCurrency = $aCurrency['sign'];
                    break;
                case 'conversion_rate':
                    $mCurrency = $aCurrency['conversion_rate'];
                    break;
                case 'format':
                    $mCurrency = $aCurrency['format'];
                    break;
                default:
                    $mCurrency = $aCurrency;
                    break;
            }
        }

        return $mCurrency;
    }


    /**
     * method returns template path
     *
     * @param string $sTemplate
     * @return string
     */
    public static function getTemplatePath($sTemplate)
    {
        return \FacebookProductAd::$oModule->getTemplatePath($sTemplate);
    }

    /**
     * method returns link object
     *
     * @return obj
     */
    public static function getLinkObj()
    {
        return \FacebookProductAd::$oContext->link;
    }

    /**
     * method returns product link
     *
     * @param obj $oProduct
     * @param int $iLangId
     * @param string $sCatRewrite
     * @return string
     */
    public static function getProductLink($oProduct, $iLangId, $sCatRewrite = '')
    {
        $sProdUrl = '';

        if (\Configuration::get('PS_REWRITING_SETTINGS')) {
            $sProdUrl = self::getLinkObj()->getProductLink($oProduct, null, null, null, (int) $iLangId, null, 0, true);
        } else {
            $sProdUrl = self::getLinkObj()->getProductLink($oProduct, null, null, null, (int) $iLangId, null, 0, false);
        }

        return $sProdUrl;
    }

    /**
     * method returns the product condition
     *
     * @param string $sCondition
     * @return string
     */
    public static function getProductCondition($sCondition = null)
    {
        $sResult = '';

        if (
            $sCondition !== null
            && in_array($sCondition, array('new', 'used', 'refurbished'))
        ) {
            $sResult = $sCondition;
        } else {
            $sResult = !empty(\FacebookProductAd::$conf['FPA_COND']) ? \FacebookProductAd::$conf['FPA_COND'] : 'new';
        }

        return $sResult;
    }

    /**
     * method returns product image
     *
     * @param obj $oProduct
     * @param string $sImageType
     * @param array $aForceImage
     * @param string $sForceDomainName
     * @return obj
     */
    public static function getProductImage(\Product &$oProduct, $sImageType = null, $aForceImage = false, $sForceDomainName = null)
    {
        $sImgUrl = '';

        if (\Validate::isLoadedObject($oProduct)) {
            // use case - get Image
            $aImage = $aForceImage !== false ? $aForceImage : $oProduct->getImages(\FacebookProductAd::$iCurrentLang);

            if (!empty($aImage)) {
                // get image url
                if ($sImageType !== null) {
                    $sImgUrl = \Context::getContext()->link->getImageLink($oProduct->link_rewrite, $oProduct->id . '-' . $aImage['id_image'], $sImageType);
                } else {
                    $sImgUrl = \Context::getContext()->link->getImageLink($oProduct->link_rewrite, $oProduct->id . '-' . $aImage);
                }
            }
        }
        return $sImgUrl;
    }

    /**
     * method truncate current request_uri in order to delete params : sAction and sType
     *
     * @param mixed: string or array $mNeedle
     * @return mixed
     */
    public static function truncateUri($mNeedle = '&sAction')
    {
        // set tmp
        $aQuery = is_array($mNeedle) ? $mNeedle : array($mNeedle);

        // get URI
        $sURI = $_SERVER['REQUEST_URI'];

        foreach ($aQuery as $sNeedle) {
            $sURI = strstr($sURI, $sNeedle) ? substr($sURI, 0, strpos($sURI, $sNeedle)) : $sURI;
        }
        return $sURI;
    }

    /**
     * method check if specific module and module's vars are available
     *
     * @param int $sModuleName
     * @param array $aCheckedVars
     * @param bool $bObjReturn
     * @param bool $bOnlyInstalled
     * @return mixed : true or false or obj
     */
    public static function isInstalled($sModuleName, array $aCheckedVars = array(), $bObjReturn = false, $bOnlyInstalled = false)
    {
        $mReturn = false;

        // use case - check module is installed in DB
        if (\Module::isInstalled($sModuleName)) {
            if (!$bOnlyInstalled) {
                $oModule = \Module::getInstanceByName($sModuleName);

                if (!empty($oModule)) {
                    // check if module is activated
                    $aActivated = \Db::getInstance()->ExecuteS('SELECT id_module as id, active FROM ' . _DB_PREFIX_ . 'module WHERE name = "' . pSQL($sModuleName) . '" AND active = 1');

                    if (!empty($aActivated[0]['active'])) {
                        $mReturn = true;

                        if (version_compare(_PS_VERSION_, '1.5', '>')) {
                            $aActivated = \Db::getInstance()->ExecuteS('SELECT * FROM ' . _DB_PREFIX_ . 'module_shop WHERE id_module = ' . (int)$aActivated[0]['id'] . ' AND id_shop = ' . (int)\Context::getContext()->shop->id);

                            if (empty($aActivated)) {
                                $mReturn = false;
                            }
                        }

                        if ($mReturn) {
                            if (!empty($aCheckedVars)) {
                                foreach ($aCheckedVars as $sVarName) {
                                    $mVar = \Configuration::get($sVarName);

                                    if (empty($mVar)) {
                                        $mReturn = false;
                                    }
                                }
                            }
                        }
                    }
                }
                if ($mReturn && $bObjReturn) {
                    $mReturn = $oModule;
                }
                unset($oModule);
            } else {
                $mReturn = true;
            }
        }
        return $mReturn;
    }

    /**
     * method write breadcrumbs of product for category
     *
     * @param int $iCatId
     * @param int $iLangId
     * @param string $sPath
     * @param bool $bEncoding
     * @return string
     */
    public static function getProductPath($iCatId, $iLangId, $sPath = '', $bEncoding = true)
    {
        $oCategory = new \Category($iCatId);

        return ((\Validate::isLoadedObject($oCategory) ? strip_tags(self::getPath((int) $oCategory->id, (int) $iLangId, $sPath, $bEncoding)) : ''));
    }

    /**
     * method write breadcrumbs of product for category
     *
     * Forced to redo the function from Tools here as it works with cookie
     * for language, not a passed parameter in the function
     *
     * @param int $iCatId
     * @param int $iLangId
     * @param string $sPath
     * @param bool $bEncoding
     * @return string
     */
    public static function getPath($iCatId, $iLangId, $sPath = '', $bEncoding = true)
    {
        $mReturn = '';

        if ($iCatId == 1) {
            $mReturn = $sPath;
        } else {
            // get pipe
            $sPipe = ' > ';
            $sFullPath = '';

            $aInterval = \Category::getInterval($iCatId);
            $aIntervalRoot = \Category::getInterval(\Context::getContext()->shop->getCategory());

            if (!empty($aInterval) && !empty($aIntervalRoot)) {
                $sQuery = 'SELECT c.id_category, cl.name, cl.link_rewrite'
                    . ' FROM ' . _DB_PREFIX_ . 'category c'
                    . \Shop::addSqlAssociation('category', 'c', false)
                    . ' LEFT JOIN ' . _DB_PREFIX_ . 'category_lang cl ON (cl.id_category = c.id_category' . \Shop::addSqlRestrictionOnLang('cl') . ')'
                    . 'WHERE c.nleft <= ' . (int)$aInterval['nleft']
                    . ' AND c.nright >= ' . (int)$aInterval['nright']
                    . ' AND c.nleft >= ' . (int)$aIntervalRoot['nleft']
                    . ' AND c.nright <= ' . (int)$aIntervalRoot['nright']
                    . ' AND cl.id_lang = ' . (int) $iLangId
                    . ' AND c.level_depth > ' . (int) $aIntervalRoot['level_depth']
                    . ' ORDER BY c.level_depth ASC';

                $aCategories = \Db::getInstance()->executeS($sQuery);

                $iCount = 1;
                $nCategories = count($aCategories);

                foreach ($aCategories as $aCategory) {
                    $sFullPath .= ($bEncoding ? htmlentities($aCategory['name'], ENT_NOQUOTES, 'UTF-8') : $aCategory['name']) . (($iCount++ != $nCategories || !empty($sPath)) ? $sPipe : '');
                }
                $mReturn = $sFullPath . $sPath;
            }
        }

        return $mReturn;
    }

    /**
     * method process categories to generate tree of them
     *
     * @param array $aCategories
     * @param array $aIndexedCat
     * @param array $aCurrentCat
     * @param int $iCurrentIndex
     * @param int $iDefaultId
     * @param bool $bFirstExec
     * @return array
     */
    public static function recursiveCategoryTree(array $aCategories, array $aIndexedCat, $aCurrentCat, $iCurrentIndex = 1, $iDefaultId = null, $bFirstExec = false)
    {
        // set variables
        static $_aTmpCat;
        static $_aFormatCat;

        if ($bFirstExec) {
            $_aTmpCat = null;
            $_aFormatCat = null;
        }

        if (!isset($_aTmpCat[$aCurrentCat['infos']['id_parent']])) {
            $_aTmpCat[$aCurrentCat['infos']['id_parent']] = 0;
        }
        $_aTmpCat[$aCurrentCat['infos']['id_parent']] += 1;

        // calculate new level
        $aCurrentCat['infos']['iNewLevel'] = $aCurrentCat['infos']['level_depth'] + 1;

        // calculate type of gif to display - displays tree in good
        $aCurrentCat['infos']['sGifType'] = (count($aCategories[$aCurrentCat['infos']['id_parent']]) == $_aTmpCat[$aCurrentCat['infos']['id_parent']] ? 'f' : 'b');

        // calculate if checked
        if (in_array($iCurrentIndex, $aIndexedCat)) {
            $aCurrentCat['infos']['bCurrent'] = true;
        } else {
            $aCurrentCat['infos']['bCurrent'] = false;
        }

        // define classname with default cat id
        $aCurrentCat['infos']['mDefaultCat'] = ($iDefaultId === null) ? 'default' : $iCurrentIndex;

        $_aFormatCat[] = $aCurrentCat['infos'];

        if (isset($aCategories[$iCurrentIndex])) {
            foreach ($aCategories[$iCurrentIndex] as $iCatId => $aCat) {
                if ($iCatId != 'infos') {
                    self::recursiveCategoryTree($aCategories, $aIndexedCat, $aCategories[$iCurrentIndex][$iCatId], $iCatId);
                }
            }
        }
        return $_aFormatCat;
    }

    /**
     * method process brands to generate tree of them
     *
     * @param array $aBrands
     * @param array $aIndexedBrands
     * @return array
     */
    public static function recursiveBrandTree(array $aBrands, array $aIndexedBrands)
    {
        // set
        $aFormatBrands = array();

        foreach ($aBrands as $iIndex => $aBrand) {
            $aFormatBrands[] = array(
                'id' => $aBrand['id_manufacturer'],
                'name' => $aBrand['name'],
                'checked' => (in_array($aBrand['id_manufacturer'], $aIndexedBrands) ? true : false)
            );
        }

        return $aFormatBrands;
    }

    /**
     * method process suppliers to generate tree of them
     *
     * @param array $aSuppliers
     * @param array $aIndexedSuppliers
     * @return array
     */
    public static function recursiveSupplierTree(array $aSuppliers, array $aIndexedSuppliers)
    {
        // set
        $aFormatSuppliers = array();

        foreach ($aSuppliers as $iIndex => $aSupplier) {
            $aFormatSuppliers[] = array(
                'id' => $aSupplier['id_supplier'],
                'name' => $aSupplier['name'],
                'checked' => (in_array($aSupplier['id_supplier'], $aIndexedSuppliers) ? true : false)
            );
        }

        return $aFormatSuppliers;
    }

    /**
     * method round on numeric
     *
     * @param float $fVal
     * @param int $iPrecision
     * @return float
     */
    public static function round($fVal, $iPrecision = 2)
    {
        if (method_exists('Tools', 'ps_round')) {
            $fVal = \Tools::ps_round($fVal, $iPrecision);
        } else {
            $fVal = round($fVal, $iPrecision);
        }

        return $fVal;
    }

    /**
     * method set host
     *
     * @return string
     */
    public static function setHost()
    {
        if (\Configuration::get('PS_SHOP_DOMAIN') != false) {
            $sURL = 'http://' . \Configuration::get('PS_SHOP_DOMAIN');
        } else {
            $sURL = 'http://' . $_SERVER['HTTP_HOST'];
        }

        return $sURL;
    }

    /**
     * method set the XML file's prefix
     *
     * @return string
     */
    public static function setXmlFilePrefix()
    {
        return 'facebookproductad' . \FacebookProductAd::$conf['FPA_FEED_TOKEN'];
    }

    /**
     * method clear all generated files
     *
     * @return bool
     */
    public static function cleanUpFiles()
    {
        foreach (\FacebookProductAd::$aAvailableLanguages as $aLanguage) {
            // get each countries by language
            $aCountries = moduleConfiguration::FPA_AVAILABLE_COUNTRIES[$aLanguage['iso_code']];

            foreach ($aCountries as $sCountry => $aLocaleData) {
                // detect file's suffix and clear file
                $fileSuffix = self::buildFileSuffix($aLanguage['iso_code'], $sCountry);
                @unlink(moduleConfiguration::FPA_SHOP_PATH_ROOT . \FacebookProductAd::$sFilePrefix . '.' . $fileSuffix . '.xml');
            }
        }
    }

    /**
     * method Build file suffix based on language and country ISO code
     *
     * @param string $sLangIso
     * @param string $sCountryIso
     * @param int $iShopId
     * @return string
     */
    public static function buildFileSuffix($sLangIso, $sCountryIso, $iShopId = 0)
    {
        if (\Tools::strtolower($sLangIso) == \Tools::strtolower($sCountryIso)) {
            $sSuffix = \Tools::strtolower($sLangIso);
        } else {
            $sSuffix = \Tools::strtolower($sLangIso) . '.' . \Tools::strtolower($sCountryIso);
        }

        $sSuffix .= ($iShopId ? '.shop' . $iShopId : '.shop' . \FacebookProductAd::$iShopId);


        return $sSuffix;
    }

    /**
     * method returns all available condition
     */
    public static function getConditionType()
    {
        return array(
            'new' => \FacebookProductAd::$oModule->l('New', 'moduleTools'),
            'used' => \FacebookProductAd::$oModule->l('Used', 'moduleTools'),
            'refurbished' => \FacebookProductAd::$oModule->l('Refurbished', 'moduleTools'),
        );
    }

    /**
     * method returns all available description
     */
    public static function getDescriptionType()
    {
        return array(
            1 => \FacebookProductAd::$oModule->l('Short description', 'moduleTools'),
            2 => \FacebookProductAd::$oModule->l('Long description', 'moduleTools'),
            3 => \FacebookProductAd::$oModule->l('Both', 'moduleTools'),
            4 => \FacebookProductAd::$oModule->l('Meta-description', 'moduleTools')
        );
    }

    /**
     * method set all available attributes managed in facebook flux
     */
    public static function loadFacebookTags()
    {
        return array(
            '_no_available_for_order' => array(
                'label' => \FacebookProductAd::$oModule->l('Product not available for order', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because the "Available for order" option is not checked', 'moduleTools') . '.',
                'faq_id' => 491,
                'anchor' => ''
            ),
            '_no_product_name' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product name', 'moduleTools'),
                'type' => 'error',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because the product names are missing', 'moduleTools') . '.',
                'faq_id' => 280,
                'anchor' => ''
            ),
            '_no_required_data' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing mandatory information', 'moduleTools'),
                'type' => 'error',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because one of this mandatory product information is missing: name / description / link / image link', 'moduleTools') . '.',
                'faq_id' => 491,
                'anchor' => ''
            ),
            '_no_export_no_supplier_ref' => array(
                'label' => \FacebookProductAd::$oModule->l('Product without MPN', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because they do not have an MPN reference', 'moduleTools') . '.',
                'faq_id' => 131,
                'anchor' => ''
            ),
            '_no_export_no_ean_upc' => array(
                'label' => \FacebookProductAd::$oModule->l('Product without GTIN code (EAN13 or UPC)', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because they do not have an EAN13/UPC reference (GTIN code)', 'moduleTools') . '.',
                'faq_id' => 131,
                'anchor' => ''
            ),
            '_no_export_no_stock' => array(
                'label' => \FacebookProductAd::$oModule->l('No stock', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because they are out of stock', 'moduleTools') . '.',
                'faq_id' => 131,
                'anchor' => ''
            ),
            '_no_export_min_price' => array(
                'label' => \FacebookProductAd::$oModule->l('Product under min price', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Products not exported because their price is lower than the minimum value defined in the configuration', 'moduleTools') . '.',
                'faq_id' => 131,
                'anchor' => ''
            ),
            //to be removed because this was only valid for gmcp old versions
            'excluded' => array(
                'label' => \FacebookProductAd::$oModule->l('Excluded product list', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('this product or combination has been excluded from your feed as you defined it in the exclusion rules tab', 'moduleTools') . '.',
                'faq_id' => 0,
                'anchor' => ''
            ),
            'id' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product ID', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "ID" tag => This is the unique identifier of the item', 'moduleTools') . '.',
                'faq_id' => 261,
                'anchor' => ''
            ),
            'title' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product title', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "TITLE" tag => This is the title of the item', 'moduleTools') . '.',
                'faq_id' => 280,
                'anchor' => ''
            ),
            'description' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product description', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "DESCRIPTION" tag => This is the description of the item', 'moduleTools') . '.',
                'faq_id' => 273,
                'anchor' => ''
            ),
            'google_product_category' => array(
                'label' => \FacebookProductAd::$oModule->l('No Facebook category', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "FACEBOOK PRODUCT CATEGORY" tag => You have to associate each product default category with an official Facebook category', 'moduleTools') . '.',
                'faq_id' => 281,
                'anchor' => ''
            ),
            'product_type' => array(
                'label' => \FacebookProductAd::$oModule->l('No product type', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "PRODUCT TYPE" tag => Unlike the "Facebook Product Category" tag, the "Product Type" tag contains the information about the category of the product according to your own classification', 'moduleTools') . '.',
                'faq_id' => 258,
                'anchor' => ''
            ),
            'link' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product link', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "LINK" tag => This is the link of the item', 'moduleTools') . '.',
                'faq_id' => 260,
                'anchor' => ''
            ),
            'image_link' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing image link', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "IMAGE LINK" tag => This is the URL of the main image of the product', 'moduleTools') . '.',
                'faq_id' => 257,
                'anchor' => ''
            ),
            'condition' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product condition', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "CONDITION" tag => This is the condition of the item. There are only 3 accepted values: "new", "refurbished" and "used"', 'moduleTools') . '.',
                'faq_id' => 259,
                'anchor' => ''
            ),
            'availability' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product availability', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "AVAILABILITY" tag => This indicates the availability of the item. There are only 4 accepted values: "in stock", "out of stock", "available for order" and "discontinued"', 'moduleTools') . '.',
                'faq_id' => 262,
                'anchor' => ''
            ),
            'price' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing product price', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "PRICE" tag => This is the price of the item', 'moduleTools') . '.',
                'faq_id' => 279,
                'anchor' => ''
            ),
            'gtin' => array(
                'label' => \FacebookProductAd::$oModule->l('No GTIN code', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "GTIN" tag => The "Global Trade Item Number" is one of the Unique Product Identifiers', 'moduleTools') . '.',
                'faq_id' => 263,
                'anchor' => ''
            ),
            'brand' => array(
                'label' => \FacebookProductAd::$oModule->l('No product brand', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "BRAND" tag => The product brand is one of the Unique Product Identifiers', 'moduleTools') . '.',
                'faq_id' => 278,
                'anchor' => ''
            ),
            'mpn' => array(
                'label' => \FacebookProductAd::$oModule->l('No MPN reference', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "MPN" tag => The "Manufacturer Part Number" is one of the Unique Product Identifiers', 'moduleTools') . '.',
                'faq_id' => 264,
                'anchor' => ''
            ),
            'adult' => array(
                'label' => \FacebookProductAd::$oModule->l('No adult tag', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "ADULT" tag => This tag indicates that the item is for adults only', 'moduleTools') . '.',
                'faq_id' => 274,
                'anchor' => ''
            ),
            'gender' => array(
                'label' => \FacebookProductAd::$oModule->l('No gender tag', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "GENDER" tag => This tag allows you specify the gender your product is designed for. You can choose between 3 predefined values: "male", "female" or "unisex"', 'moduleTools') . '.',
                'faq_id' => 270,
                'anchor' => ''
            ),
            'age_group' => array(
                'label' => \FacebookProductAd::$oModule->l('No age group tag', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "AGE GROUP" tag => This tag allows you to specify the age group your product is designed for. You can choose between 7 predefined values: "adults", "all ages", "teens", "kids", "toddlers", "infants" or "newborns"', 'moduleTools') . '.',
                'faq_id' => 271,
                'anchor' => ''
            ),
            'color' => array(
                'label' => \FacebookProductAd::$oModule->l('No color', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "COLOR" tag => This tag indicates the color of the item', 'moduleTools') . '.',
                'faq_id' => 276,
                'anchor' => ''
            ),
            'size' => array(
                'label' => \FacebookProductAd::$oModule->l('No size', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "SIZE" tag => This tag indicates the size of the item', 'moduleTools') . '.',
                'faq_id' => 275,
                'anchor' => ''
            ),
            'material' => array(
                'label' => \FacebookProductAd::$oModule->l('No material tag', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "MATERIAL" tag => This tag indicates the material the item is made from', 'moduleTools') . '.',
                'faq_id' => 268,
                'anchor' => ''
            ),
            'pattern' => array(
                'label' => \FacebookProductAd::$oModule->l('No pattern tag', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "PATTERN" tag => This tag indicates the pattern or graphic print on the item', 'moduleTools') . '.',
                'faq_id' => 269,
                'anchor' => ''
            ),
            'item_group_id' => array(
                'label' => \FacebookProductAd::$oModule->l('No item group id', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "ITEM GROUP ID" tag => All items that are variants of a same product must have the same item group id', 'moduleTools') . '.',
                'faq_id' => 491,
                'anchor' => ''
            ),
            'shipping_weight' => array(
                'label' => \FacebookProductAd::$oModule->l('No shipping weight tag', 'moduleTools'),
                'type' => 'warning',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('The "SHIPPING WEIGHT" tag => This is the weight of the item used to calculate the shipping cost of the item. The shipping weight of the item must be in lb, oz, g, or kg.', 'moduleTools') . '.',
                'faq_id' => 282,
                'anchor' => ''
            ),
            'shipping' => array(
                'label' => \FacebookProductAd::$oModule->l('Missing shipping information', 'moduleTools'),
                'type' => 'error',
                'mandatory' => true,
                'msg' => \FacebookProductAd::$oModule->l('The "SHIPPING" tag => The shipping tag lets you indicates the shipping details for the item', 'moduleTools') . '.',
                'faq_id' => 51,
                'anchor' => ''
            ),
            'title_length' => array(
                'label' => \FacebookProductAd::$oModule->l('Too long title', 'moduleTools'),
                'type' => 'notice',
                'mandatory' => false,
                'msg' => \FacebookProductAd::$oModule->l('Facebook requires your product titles to be no more than 150 characters long', 'moduleTools') . '.',
                'faq_id' => 280,
                'anchor' => ''
            ),
        );
    }

    /**
     * method returns the Facebook taxonomy file's content
     *
     * @param string $sUrl
     * @return string
     */
    public static function getFacebookFile($sUrl)
    {
        $sContent = false;

        // Let's try first with file_get_contents
        if (ini_get('allow_url_fopen')) {
            $sContent = (method_exists('Tools', 'file_get_contents') ? \Tools::file_get_contents($sUrl) : file_get_contents($sUrl));
        }

        // Returns false ? Try with CURL if available
        if ($sContent === false && function_exists('curl_init')) {
            $ch = curl_init();

            curl_setopt_array($ch, array(
                CURLOPT_URL => $sUrl,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_VERBOSE => true
            ));

            $sContent = @curl_exec($ch);
            curl_close($ch);
        }

        // Will return false if no method is available, or if either fails
        // This will cause a JavaScript alert to be triggered by the AJAX call
        return $sContent;
    }

    /**
     * method returns the generated report files
     *
     * @return array
     */
    public static function getGeneratedReport()
    {
        $reporting_output = array();
        $reportingList = Reporting::getReportingList(\FacebookProductAd::$iShopId);

        if (!empty($reportingList)) {
            foreach ($reportingList as $list) {
                $reporting_data = explode('_', $list['iso_feed']);

                $id_lang = \Language::getIdByIso($reporting_data[0]);
                $language = new \Language((int)$id_lang);

                $id_currency = \Currency::getIdByIsoCode($reporting_data[2]);
                $currency = new \Currency($id_currency);

                $id_country = \Country::getByIso(\Tools::strtolower($reporting_data[1]));
                $country = new \Country((int)$id_country);

                $reporting_output[] = array(
                    'full' => $reporting_data[0] . '_' . $reporting_data[1] . '_' . $reporting_data[2],
                    'lang_iso' => $language->name . ' - ' . \Tools::strtoupper($language->iso_code),
                    'currency' => $currency->sign . ' - ' . $currency->iso_code,
                    'country' => \Country::getNameById(\FacebookProductAd::$iCurrentLang, $country->id) . ' - ' . $country->iso_code,
                );
            }
        }

        return $reporting_output;
    }

    /**
     * method format the product title by uncap or not or leave uppercase only first character of each word
     *
     * @param string $sTitle
     * @param int $iFormatMode
     * @return string
     */
    public static function formatProductTitle($sTitle, $iFormatMode = 0)
    {
        $sResult = '';

        // format title
        if ($iFormatMode == 0) {
            $sResult = self::strToUtf8($sTitle);
        } else {
            $sResult = self::strToLowerUtf8($sTitle);

            if ($iFormatMode == 1) {
                $aResult = explode(' ', $sResult);

                foreach ($aResult as &$sWord) {
                    $sWord = \Tools::ucfirst(trim($sWord));
                }

                $sResult = implode(' ', $aResult);
            } else {
                $sResult = \Tools::ucfirst(trim($sResult));
            }
        }

        return $sResult;
    }

    /**
     * method format the product name with combination
     *
     * @param int $iAttrId
     * @param int $iCurrentLang
     * @param int $iShopId
     * @return string
     */
    public static function getProductCombinationName($iAttrId, $iCurrentLang, $iShopId)
    {
        // set var
        $sProductName = '';

        $aCombinations = moduleDao::getProductComboAttributes($iAttrId, $iCurrentLang, $iShopId);

        if (!empty($aCombinations)) {
            $sExtraName = '';
            foreach ($aCombinations as $c) {
                $sExtraName .= ' ' . \Tools::stripslashes($c['name']);
            }
            $sProductName .= $sExtraName;
            unset($sExtraName);
        }
        unset($aCombinations);

        return $sProductName;
    }


    /**
     * method uncap the product title
     *
     * @param int $iAdvancedProdName
     * @param string $sProdName
     * @param string $sCatName
     * @param string $sManufacturerName
     * @param int $iLength
     * @return string
     */
    public static function truncateProductTitle($iAdvancedProdName, $sProdName, $sCatName, $sManufacturerName, $iLength)
    {
        if (function_exists('mb_substr')) {
            switch ($iAdvancedProdName) {
                case 0:
                    $sProdName = mb_substr($sProdName, 0, $iLength);
                    break;
                case 1:
                    $sProdName = mb_substr($sCatName . ' - ' . $sProdName, 0, $iLength);
                    break;
                case 2:
                    $sProdName = mb_substr($sProdName . ' - ' . $sCatName, 0, $iLength);
                    break;
                case 3:
                    $sBrand = !empty($sManufacturerName) ? $sManufacturerName . ' - ' : '';
                    $sProdName = mb_substr($sBrand . $sProdName, 0, $iLength);
                    break;
                case 4:
                    $sBrand = !empty($sManufacturerName) ? ' - ' . $sManufacturerName : '';
                    $sProdName = mb_substr($sProdName . $sBrand, 0, $iLength);
                    break;
                default:
                    break;
            }
        }

        return \Tools::stripslashes($sProdName);
    }

    /**
     * method Used by uncapProductTitle. strtolower doesn't work with UTF-8
     * The second solution if no mb_strtolower available is not perfect but will work
     * with most European languages. Worse comes to worse, the person may chose not to uncap
     *
     * @param $sString
     * return string
     */
    public static function strToLowerUtf8($sString)
    {
        return ((function_exists('mb_strtolower') ? mb_strtolower($sString, 'utf-8') : utf8_encode(\Tools::strtolower(utf8_decode($sString)))));
    }

    /**
     * method Used by uncapProductTitle. strToUtf8 doesn't work with UTF-8
     * The second solution if no mb_convert_encoding available is not perfect but will work
     * with most European languages. Worse comes to worse, the person may chose not to uncap
     *
     * @param $sString
     * return string
     */
    public static function strToUtf8($sString)
    {
        return ((function_exists('mb_convert_encoding') ? mb_convert_encoding($sString, 'utf-8') : utf8_encode(utf8_decode($sString))));
    }

    /**
     * method Check file based on language and country ISO code
     *
     * @param string $sIsoLang
     * @param string $sIsoCountry
     * @param string $sCurrency
     * @return bool
     */
    public static function checkReportFile($sIsoLang, $sIsoCountry, $sCurrency)
    {
        $sFilename = moduleConfiguration::FPA_REPORTING_DIR . 'reporting-' . $sIsoLang . '-' . \Tools::strtolower($sIsoCountry) . '-' . $sCurrency . '.txt';

        return (
            (file_exists($sFilename) && filesize($sFilename)) ? true : false);
    }

    /**
     * method returns price by considering the merchant option in the back office
     * @param array $aParams
     * @param bool $bUseTax
     * @param bool $bUseShippings
     * @param bool $bUseWrapping
     * @return float
     */
    public static function getOrderPrice($aParams, $bUseTax, $bUseShipping, $bUseWrapping)
    {
        $fOderAmount = 0.0;

        if (!empty($aParams)) {
            //case with tax
            if (!empty($bUseTax)) {
                if (!empty($bUseShipping) && !empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_paid;
                } elseif (empty($bUseShipping) && !empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_paid - $aParams->total_shipping_tax_incl;
                } elseif (!empty($bUseShipping) && empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_paid - $aParams->total_wrapping_tax_incl;
                } elseif (empty($bUseShipping) && empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_paid - $aParams->total_wrapping_tax_incl - $aParams->total_shipping_tax_incl;
                }
            } //case without tax
            elseif (empty($bUseTax)) {
                if (!empty($bUseShipping) && !empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_paid_tax_excl;
                } elseif (empty($bUseShipping) && !empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_products + $aParams->total_wrapping_tax_excl;
                } elseif (!empty($bUseShipping) && empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_products + $aParams->total_shipping_tax_excl;
                } elseif (empty($bUseShipping) && empty($bUseWrapping)) {
                    $fOderAmount = $aParams->total_products;
                }
            }
        }

        return $fOderAmount;
    }

    /* Clean up MS Word style quotes and other characters Facebook does not like */
    /**
     *  method clean up MS Word style quotes and other characters Facebook does not like
     *
     * @param string $str
     * @return string
     */
    public static function cleanUp($str)
    {
        $str = str_replace('<br>', "\n", $str);
        $str = str_replace('<br />', "\n", $str);
        $str = str_replace('</p>', "\n", $str);
        $str = str_replace('<p>', '', $str);

        $quotes = array(
            "\xC2\xAB" => '"', // « (U+00AB) in UTF-8
            "\xC2\xBB" => '"', // » (U+00BB) in UTF-8
            "\xE2\x80\x98" => "'", // ‘ (U+2018) in UTF-8
            "\xE2\x80\x99" => "'", // ’ (U+2019) in UTF-8
            "\xE2\x80\x9A" => "'", // ‚ (U+201A) in UTF-8
            "\xE2\x80\x9B" => "'", // ‛ (U+201B) in UTF-8
            "\xE2\x80\x9C" => '"', // “ (U+201C) in UTF-8
            "\xE2\x80\x9D" => '"', // ” (U+201D) in UTF-8
            "\xE2\x80\x9E" => '"', // „ (U+201E) in UTF-8
            "\xE2\x80\x9F" => '"', // ‟ (U+201F) in UTF-8
            "\xE2\x80\xB9" => "'", // ‹ (U+2039) in UTF-8
            "\xE2\x80\xBA" => "'", // › (U+203A) in UTF-8
            "\xE2\x80\x94" => '-', // —
        );

        $str = strtr($str, $quotes);
        return trim(strip_tags($str));
    }

    /**
     * detectCurrentPage() method returns current page type
     */
    public static function detectCurrentPage()
    {
        $step_id = self::getStepId((int)\Context::getContext()->cart->id);

        // use case - home page
        if (\Tools::getValue('controller') == 'index') {
            $sCurrentTypePage = 'home';
        } // use case - search results page
        elseif (\Tools::getValue('controller') == 'search' && empty(\Context::getContext()->controller->module)) {
            $sCurrentTypePage = 'search';
        } // use case - order page
        elseif ((\Tools::getValue('controller') == 'order'
                || \Tools::getValue('controller') == 'orderopc')
            && \Tools::getValue('step') == false
        ) {
            if (isset(\Context::getContext()->controller->page_name)) {
                if ($step_id == 1) {
                    $sCurrentTypePage = 'checkout';
                } else {
                    if ($step_id == 0 && \Context::getContext()->controller->page_name != 'checkout') {
                        $sCurrentTypePage = 'cart';
                    } else {
                        $sCurrentTypePage = 'other';
                    }
                }
            } else {
                if ($step_id == 0) {
                    $sCurrentTypePage = 'cart';
                }
            }
        } elseif ((\Tools::getValue('controller') == 'submit') || \Tools::getValue('controller') == 'orderconfirmation') {
            $sCurrentTypePage = 'purchase';
        } // use case - category page
        elseif (\Tools::getValue('id_category')) {
            $sCurrentTypePage = 'category';
        } // use case - product page
        elseif (\Tools::getValue('id_product')) {
            $sCurrentTypePage = 'product';
        } elseif (\Tools::getValue('controller') == 'manufacturer') {
            $sCurrentTypePage = 'manufacturer';
        } elseif (\Tools::getValue('controller') == 'pricesdrop') {
            $sCurrentTypePage = 'promotion';
        } elseif (\Tools::getValue('controller') == 'newproducts') {
            $sCurrentTypePage = 'newproducts';
        } elseif (\Tools::getValue('controller') == 'bestsales') {
            $sCurrentTypePage = 'bestsales';
        } elseif (\Tools::getValue('controller') == 'cart') {
            $sCurrentTypePage = 'cart';
        } elseif (\Tools::getValue('controller') == 'contact') {
            $sCurrentTypePage = 'contact';
        } else {
            $sCurrentTypePage = 'other';
        }

        return $sCurrentTypePage;
    }

    /**
     * method returns all products displayed by home product tabs module
     *
     * @param string $sName
     * @param int $iLimit
     * @return array $aProducts
     */
    public static function getHomeProductTabs($sName, $iLimit)
    {
        // set var
        $aProducts = array();
        $aProductIds = array();

        // get the home object
        $oHome = new \BT_HookHome('home');

        $aData = $oHome->display(array());

        if (!empty($aData['assign']['aCategories'])) {
            foreach ($aData['assign']['aCategories'] as $aCategory) {
                if (!empty($aCategory['aProducts'])) {
                    $aProducts = array_merge($aProducts, $aCategory['aProducts']);
                }
            }
        }

        foreach ($aProducts as $id => $aProduct) {
            if (!isset($aProductIds[$aProduct['id_product']])) {
                $aProductIds[$aProduct['id_product']] = $aProduct;
            }
        }
        unset($aProducts);

        return $aProductIds;
    }

    /**
     * method returns all products displayed by home featured module
     *
     * @param string $sName
     * @param int $iLimit
     * @return array $aProducts
     */
    public static function getHomeFeaturedProducts($sName, $iLimit)
    {
        $oCategory = new \Category(\Context::getContext()->shop->getCategory(), \FacebookProductAd::$iCurrentLang);

        return ($oCategory->getProducts((int) \FacebookProductAd::$iCurrentLang, 1, $iLimit, 'position'));
    }

    /**
     * getNewProducts() method returns all products displayed by block new products module
     *
     * @param string $sName
     * @param int $iLimit
     * @return array $aProducts
     */
    public static function getNewProducts($sName, $iLimit)
    {
        // set var
        $aProducts = array();

        if (!\Configuration::get('PS_NB_DAYS_NEW_PRODUCT') || \Configuration::get('PS_BLOCK_NEWPRODUCTS_DISPLAY')) {
            $aProducts = \Product::getNewProducts((int) \FacebookProductAd::$iCurrentLang, 0, $iLimit);
        }

        return $aProducts;
    }

    /**
     * getBestSellersProducts() method returns all products displayed by block best sellers module
     *
     * @return array $aProducts
     */
    public static function getBestSellersProducts()
    {
        // set var
        $aProducts = array();

        if (!\Configuration::get('PS_CATALOG_MODE') || \Configuration::get('PS_BLOCK_BESTSELLERS_DISPLAY')) {
            $iProducPerPage = !empty(\Configuration::get('PS_BLOCK_BESTSELLERS_TO_DISPLAY'))  ? \Configuration::get('PS_BLOCK_BESTSELLERS_TO_DISPLAY') : 10;
            $aProducts = \ProductSale::getBestSales((int) \FacebookProductAd::$iCurrentLang, 0,  $iProducPerPage);
        }

        return $aProducts;
    }

    /**
     * method returns all products displayed by block specials module
     *
     * @param string $sName
     * @param int $iLimit
     * @return array $aProducts
     */
    public static function getBlockSpecials($sName, $iLimit)
    {
        // set var
        $aProducts = array();

        if (!\Configuration::get('BLOCKSPECIALS_SPECIALS_NBR') || \Configuration::get('PS_BLOCK_SPECIALS_DISPLAY')) {
            $aProducts = Product::getPricesDrop((int) \FacebookProductAd::$iCurrentLang, 0, $iLimit);
        }

        return $aProducts;
    }

    /**
     * Sanitize product properties formatted as array instead of a string matching to the current language
     * @param $property
     * @param $iLangId
     * @return mixed|string
     */
    public static function sanitizeProductProperty($property, $iLangId)
    {
        $content = '';

        // check if the product name is an array
        if (is_array($property)) {
            if (count($property) == 1) {
                $content = reset($property);
            } elseif (isset($property[$iLangId])) {
                $content = $property[$iLangId];
            }
        } else {
            $content = $property;
        }
        return $content;
    }

    /**
     * Handle the excluded word of the title
     * @param $product_name
     * @return mixed|string
     */
    public static function handleExcludedWords($product_name)
    {
        $excluded_words = json_decode(\FacebookProductAd::$conf['FPA_EXCLUDED_WORDS'], true);
        $product_name_clean = $product_name;

        if (!empty($excluded_words) && is_array($excluded_words)) {
            foreach ($excluded_words as $word) {
                $product_name_clean  = str_replace($word, '', $product_name_clean);
                $product_name_clean  = str_replace('  ', ' ', $product_name_clean);
            }
        }

        return $product_name_clean;
    }

    /**
     * build the display tag
     *
     * @param array $aDynTags
     * @param array $sPageType
     * @return array $aAssign all tag information
     * @throws
     */
    public static function buildDynDisplayTag($aDynTags, $sPageType)
    {
        try {
            //get the pixel information
            $oTagsCtrl = basePixel::get($sPageType, $aDynTags);

            //set Pixel code
            $aAssign['sPixel'] = \FacebookProductAd::$conf['FPA_PIXEL'];

            $oTagsCtrl->set();
            $aAssign['aDynTags'] = $oTagsCtrl->display();

            // use case - check if not empty and get the tracking type as we need it to open the fbq Facebook JS object
            if (!empty($aAssign['aDynTags'])) {

                $aAssign['sCR'] = "\n";

                $aAssign['aTrackingType'] = $aAssign['aDynTags']['tracking_type'];

                $aAssign['sJsObjName'] = moduleConfiguration::FPA_JS_NAME;
                unset($aAssign['aDynTags']['tracking_type']);
                unset($aAssign['aDynTags']['js_code']);
            }

            unset($oTagsCtrl);
        } catch (\Exception $e) {
            throw new \Exception($e->getMessage(), $e->getCode());
        }

        return $aAssign;
    }

    /**
     * get order if for Paypox
     *
     * @param int id_cart
     */
    public static function getOrderIdForPaypox($id_cart)
    {
        $iOrderId = moduleDao::getOrderIdFromCart($id_cart);

        return $iOrderId;
    }

    /**
     * check the gtin value
     *
     * @param string $sPriority the priority
     * @param array $aProduct the product information
     * @return string
     */
    public static function getGtin($sPriority, $aProduct)
    {
        $sGtin = '';

        if ($sPriority == 'ean') {
            if (
                !empty($aProduct['ean13'])
                && (\Tools::strlen($aProduct['ean13']) == 8
                    || \Tools::strlen($aProduct['ean13']) == 12
                    || \Tools::strlen($aProduct['ean13']) == 13)
            ) {
                $sGtin = $aProduct['ean13'];
            } elseif (
                !empty($aProduct['upc'])
                && (\Tools::strlen($aProduct['upc']) == 8
                    || \Tools::strlen($aProduct['upc']) == 12
                    || \Tools::strlen($aProduct['upc']) == 13)
            ) {
                $sGtin = $aProduct['upc'];
            }
        } else {
            if (
                !empty($aProduct['upc'])
                && (\Tools::strlen($aProduct['upc']) == 8
                    || \Tools::strlen($aProduct['upc']) == 12
                    || \Tools::strlen($aProduct['upc']) == 13)
            ) {
                $sGtin = $aProduct['upc'];
            } elseif (
                !empty($aProduct['ean13'])
                && (\Tools::strlen($aProduct['ean13']) == 8
                    || \Tools::strlen($aProduct['ean13']) == 12
                    || \Tools::strlen($aProduct['ean13']) == 13)
            ) {
                $sGtin = $aProduct['ean13'];
            }
        }

        return $sGtin;
    }

    /**
     * get the exclusion rules names
     *
     * @param array $aExclusionRules the rules
     *
     * @return bool
     */
    public static function getExclusionRulesName($aExclusionRules)
    {
        // Array to format th;e values with good value
        $aData = $aExclusionRules;

        foreach ($aExclusionRules as $sKey => $sValue) {
            $aTmpData = unserialize($sValue['exclusion_value']);

            if ($sValue['type'] !== null) {
                switch ($sValue['type']) {
                    case 'word':
                        if (isset($aTmpData['exclusionData'])) {
                            $aData[$sKey]['exclusion_value_text'] = $aTmpData['exclusionData'];
                        }
                        break;
                    case 'feature':
                        $aFeature = \FeatureValue::getFeatureValuesWithLang(
                            \FacebookProductAd::$iCurrentLang,
                            (int) $aTmpData['exclusionOn']
                        );
                        foreach ($aFeature as $sFeature) {
                            if (
                                $sFeature['id_feature_value'] == (int) $aTmpData['exclusionData']
                            ) {
                                $aData[$sKey]['exclusion_value_text'] = $sFeature['value'];
                            }
                        }

                        break;
                    case 'attribute':
                        $aAttribute = \AttributeGroup::getAttributes(
                            \FacebookProductAd::$iCurrentLang,
                            (int) $aTmpData['exclusionOn']
                        );

                        foreach ($aAttribute as $sAttribute) {
                            if (
                                $sAttribute['id_attribute'] == (int) $aTmpData['exclusionData']
                            ) {
                                $aData[$sKey]['exclusion_value_text'] = $sAttribute['name'];
                            }
                        }
                        break;
                    default:
                        $sType = '';
                        break;
                }
                unset($aTmpData);
                unset($aFeature);
                unset($aAttribute);
            }
        }

        return $aData;
    }

    /**
     * to compare date
     *
     * @param string $sDate1
     * @param string $sDate2
     * return int : difference entre les dates
     */
    public static function dateCompare($sDate1, $sDate2)
    {
        $dDate1 = date_create($sDate1);
        $dDate2 = date_create($sDate2);
        $iDiff = date_diff($dDate1, $dDate2);

        // if date2 > date1 return 0 else return 1
        return $iDiff->invert;
    }

    /**
     * Get the order id by its cart id.
     *
     * @param int $id_cart Cart id
     *
     * @return int $id_order
     */
    public static function getIdByCartId($id_cart)
    {
        $query = new \DbQuery();
        $query->select('o.id_order');
        $query->from('orders', 'o');
        $query->where('o.`id_cart` = ' . (int)$id_cart);

        return \Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);
    }

    /**
     * format the date for Google prerequisistes
     *
     * @param string $str
     * @return string
     */
    public static function formatDateISO8601($sDate)
    {
        $sDate = new \DateTime($sDate);

        return $sDate->format(\DateTime::ISO8601);
    }

    /**
     * method returns the consent status
     * @return int
     */
    public static function getConsentStatus()
    {
        $iConsentLvl = 0;

        // Use case with ACB module
        if (!empty(self::isInstalled('pm_advancedcookiebanner'))) {
            $iConsentLvl = \AcbCookie::getConsentLevel();
        } else {
            // Use case with the trigger click event on accept all button
            $iConsentLvl = isset(\Context::getContext()->cookie->bt_fbda_consent_lvl) ? (int)\Context::getContext()->cookie->bt_fbda_consent_lvl : 0;
        }

        return $iConsentLvl;
    }

    /**
     * method setup the customer data for API call
     *
     * @param object $context
     * @return array
     */
    public static function getApiUserData(\Context $context)
    {
        $customer_data = array();

        // Use case if the custoemer is logged we sent the value otherwise we add fake date because some parameters are mandatory on Facebook API call
        if (!empty($context->customer->id)) {

            $gender = new \Gender((int)$context->customer->id_gender);
            $gender = $gender->type == 0 ? 'm' : 'f';
            $id_address = \Address::getFirstCustomerAddressId((int)$context->customer->id);
            $customer_address = new \Address((int)$id_address);
            $state = new \State((int)$customer_address->id_state);
            $country = new \Country((int)$customer_address->id_country);

            $customer_data['event_id'] = isset($_COOKIE['PHPSESSID']) ? hash('md5', $_COOKIE['PHPSESSID']) : rand(0, 9999);
            $customer_data['em'] = hash('sha256', (string)$context->customer->email);
            $customer_data['ph'] = hash('sha256', $customer_address->phone);
            $customer_data['fn'] = hash('sha256', (string)$context->customer->firstname);
            $customer_data['ln'] = hash('sha256', (string)$context->customer->lastname);
            $customer_data['db'] = hash('sha256', (string)$context->customer->birthday);
            $customer_data['ge'] = hash('sha256', (string)$gender);
            $customer_data['zp'] = hash('sha256', (string)$customer_address->postcode);
            $customer_data['country'] = hash('sha256', (string)\Tools::strtolower($country->iso_code));
            $customer_data['external_id'] = hash('sha256', (int)$context->customer->id);
            $customer_data['client_user_agent'] = (string)$_SERVER['HTTP_USER_AGENT'];

            // If we don't activate the option about API warning
            if (empty(\FacebookProductAd::$conf['FPA_HAS_WARNING'])) {
                $customer_data['ct'] = hash('sha256', (string)$customer_address->city);
                $customer_data['st'] = hash('sha256', (string)\Tools::strtolower($state->iso_code));
                $customer_data['client_ip_address'] = (string)\Tools::getRemoteAddr();
            }
        } else {
            $customer_data['event_id'] = isset($_COOKIE['PHPSESSID']) ? hash('md5', $_COOKIE['PHPSESSID']) : rand(0, 9999);
            $customer_data['em'] = hash('sha256', '');
            $customer_data['ph'] = hash('sha256', '');
            $customer_data['fn'] = hash('sha256', '');
            $customer_data['ln'] = hash('sha256', '');
            $customer_data['db'] = hash('sha256', '');
            $customer_data['ge'] = hash('sha256', '');
            $customer_data['zp'] = hash('sha256', '');
            $customer_data['country'] = hash('sha256', '');
            $customer_data['external_id'] = hash('sha256', $context->cart->id_guest);
            $customer_data['client_user_agent'] = (string)$_SERVER['HTTP_USER_AGENT'];

            // If we don't activate the option about API warning
            if (empty(\FacebookProductAd::$conf['FPA_HAS_WARNING'])) {
                $customer_data['ct'] = hash('sha256', '');
                $customer_data['st'] = hash('sha256', '');
                $customer_data['client_ip_address'] = (string)\Tools::getRemoteAddr();
            }
        }

        return $customer_data;
    }


    /**
     * method setup the customer data for API call
     *
     * @param object $context
     * @return array
     */
    public static function getAdvancedMatchingData(\Context $context)
    {
        $customer_data = array();

        if (!empty($context->customer->id)) {
            $customer_data['em'] = (string)$context->customer->email;
            $customer_data['fn'] = (string)$context->customer->firstname;
            $customer_data['ln'] = (string)$context->customer->lastname;
        }

        return $customer_data;
    }

    /**
     * getStepId() method returns the good matching list name according to the current controller name
     *
     * @param int $iCartId
     * @return int
     */
    public static function getStepId($iCartId = 0)
    {
        $iStepId = 0;

        if ($iCartId != 0) {
            $oCheckout = moduleDao::getCartSteps($iCartId);

            if (!empty($oCheckout)) {
                // detect the personal information - step 1
                if (
                    isset($oCheckout['checkout-personal-information-step'])
                    && (isset($oCheckout['checkout-personal-information-step']->step_is_reachable)
                        && $oCheckout['checkout-personal-information-step']->step_is_reachable == 1)
                    && (isset($oCheckout['checkout-personal-information-step']->step_is_complete)
                        && $oCheckout['checkout-personal-information-step']->step_is_complete == 0)
                ) {
                    $iStepId = 0;
                }

                // detect the address information - step 2
                if (
                    isset($oCheckout['checkout-addresses-step'])
                    && (isset($oCheckout['checkout-addresses-step']->step_is_reachable)
                        && $oCheckout['checkout-addresses-step']->step_is_reachable == true)
                    && (isset($oCheckout['checkout-addresses-step']->step_is_complete)
                        && $oCheckout['checkout-addresses-step']->step_is_complete == false)
                ) {
                    $iStepId = 1;
                }
                // detect the delivery information - step 3
                if (
                    isset($oCheckout['checkout-delivery-step'])
                    && (isset($oCheckout['checkout-delivery-step']->step_is_reachable)
                        && $oCheckout['checkout-delivery-step']->step_is_reachable == 1)
                    && (isset($oCheckout['checkout-delivery-step']->step_is_complete)
                        && $oCheckout['checkout-delivery-step']->step_is_complete == 0)
                ) {
                    $iStepId = 2;
                }
                // detect the payment information - step 4
                if (
                    isset($oCheckout['checkout-payment-step'])
                    && (isset($oCheckout['checkout-payment-step']->step_is_reachable)
                        && $oCheckout['checkout-payment-step']->step_is_reachable == 1)
                    && (isset($oCheckout['checkout-payment-step']->step_is_complete)
                        && $oCheckout['checkout-payment-step']->step_is_complete == 0)
                ) {
                    $iStepId = 3;
                }
            }
        }

        return $iStepId;
    }

    /**
     * method return available countries supported by Facebook
     *
     * @return array
     */
    public static function getAvailableTaxonomyCountries()
    {
        $saved_taxonomies = Feeds::getSavedTaxonomies((int)\FacebookProductAd::$iShopId);
        $shop_countries = \Country::getCountries((int) \FacebookProductAd::$oContext->cookie->id_lang, true);
        $taxonomies_output = array();

        if (!empty($saved_taxonomies)) {
            foreach ($saved_taxonomies as $data) {
                $id_country = \Country::getByIso(\Tools::strtolower($data['iso_country']));
                if (isset($shop_countries[$id_country])) {
                    $country = new \Country($id_country);
                    $taxonomies_output[$data['taxonomy']]['countries'][] = isset($country->name[\FacebookProductAd::$oContext->cookie->id_lang]) ? $country->name[\FacebookProductAd::$oContext->cookie->id_lang] : '';
                    $taxonomies_output[$data['taxonomy']]['id_lang'] = 1;
                }
            }
        }

        foreach ($taxonomies_output as $key => $data_output) {
            if (!empty($data_output['countries'])) {
                $taxonomies_output[$key]['countries'] = array_unique($data_output['countries']);
            }
        }

        return $taxonomies_output;
    }

    /**
     * method returns available carriers for one country zone
     *
     * @param int $iCountryZone
     * @return array
     */
    public static function getAvailableCarriers($iCountryZone)
    {
        $aCarriers = \Carrier::getCarriers((int) \FacebookProductAd::$oContext->cookie->id_lang, true, false, (int) $iCountryZone, null, 5);

        return $aCarriers;
    }

    /**
     * method to build and factorize contentIds
     *
     * @param $pageType
     * @param $isoLang
     * @param $productId
     * @param $productsIds
     * @param $quote
     * @param $openTag
     * @param $closeTag
     *
     * @return string|array
     */
    public static function buildContentIds($pageType, $isoLang = null, $productId = null, $productsIds = array(), $quote = '\'', $openTag = '[', $closeTag = ']')
    {
        //init vars to create pixel
        $langPrefix = '';
        $outpoutContentIds = '';
        $prefixId = \Tools::strtoupper(\FacebookProductAd::$conf['FPA_ID_PREFIX']);
        $separator = \FacebookProductAd::$conf['FPA_COMBO_SEPARATOR'];
        $combo = \FacebookProductAd::$conf['FPA_P_COMBOS'];

        if (!empty(\FacebookProductAd::$conf['FPA_ADD_LANG_ID']) && !empty($isoLang)) {
            $langPrefix = !empty($isoLang) ? \Tools::strtoupper($isoLang) : '';
        }
        // behaviour on page product
        if ($pageType == 'product') {
            if (empty($combo)) {
                // if product is not a combination
                $outpoutContentIds = $prefixId . $langPrefix . $productId;
            } else {
                // if there is combination
                $defaultProductComboId = \Product::getDefaultAttribute($productId);
                if (!empty($defaultProductComboId)) {
                    $outpoutContentIds = $prefixId . $langPrefix . $productId . $separator . $defaultProductComboId;
                } else {
                    $outpoutContentIds = $prefixId . $langPrefix . $productId;
                }
            }
        }
        // behaviour on page lists
        else if ($pageType == 'product_listing') {
            $productsData = array();
            if (!empty($productsIds)) {
                // reset productsIds to create table of productsIds
                reset($productsIds);
                foreach ($productsIds as $productsId) {
                    if (empty($combo)) {
                        $productsData[] = $quote . $prefixId . $langPrefix . $productsId['id_product'] . $quote;
                    } else {
                        // Use case for the default combo and get the catalog matching
                        $defaultProductComboId = \Product::getDefaultAttribute($productsId['id_product']);

                        if (!empty($defaultProductComboId)) {
                            $productsData[] = $quote . $prefixId . $langPrefix . $productsId['id_product'] . $separator . $defaultProductComboId . $quote;
                        } else {
                            $productsData[] = $quote . $prefixId . $langPrefix . $productsId['id_product'] . $quote;
                        }
                    }
                }
                array_unique($productsData);

                //init the string
                $outpoutContentIds = $openTag;
                $outpoutContentIds .= implode(',', $productsData);
                $outpoutContentIds .= $closeTag;
            }
        }
        // send return to the method
        return $outpoutContentIds;
    }
}
