<?php
/**
 * Dynamic Ads + Pixel
 *
 * @author    businesstech.fr <modules@businesstech.fr> - https://www.businesstech.fr/
 * @copyright Business Tech - https://www.businesstech.fr/
 * @license   see file: LICENSE.txt
 *
 *           ____    _______
 *          |  _ \  |__   __|
 *          | |_) |    | |
 *          |  _ <     | |
 *          | |_) |    | |
 *          |____/     |_|
 */

namespace FacebookProductAd\Xml;

use FacebookProductAd\ModuleLib\moduleTools;
use FacebookProductAd\ModuleLib\moduleReporting;
use FacebookProductAd\Dao\moduleDao;
use FacebookProductAd\Configuration\moduleConfiguration;

class xmlCombination extends baseXml
{
    /**
     * Magic Method __construct
     *
     * @param array $aParams
     */
    public function __construct(array $aParams = null)
    {
        parent::__construct($aParams);
    }

    /**
     * Magic Method __destruct
     */
    public function __destruct()
    {
    }

    /**
     * hasCombination() method load products combination
     *
     * @param int $iShopId
     * @param int $iProductId
     * @param bool $bHasAttributes
     * @return array
     */
    public function hasCombination($iProductId, $bExcludedProduct = false)
    {
        return moduleDao::getProductCombination($this->aParams['iShopId'], $iProductId, $bExcludedProduct);
    }

    /**
     * buildDetailProductXml() method build product XML tags
     *
     * @return mixed
     */
    public function buildDetailProductXml()
    {
        // set the product ID
        $this->data->step->id = $this->data->p->id . \FacebookProductAd::$conf['FPA_COMBO_SEPARATOR'] .  $this->data->c['id_product_attribute'];
        $this->data->step->id_no_combo = $this->data->p->id;
        $id_lang = !empty((int)\Tools::getValue('fpa_lang_id')) ? (int)\Tools::getValue('fpa_lang_id') : (int)\Tools::getValue('iLangId');
        $id_shop = !empty((int)\Tools::getValue('id_shop')) ? (int)\Tools::getValue('id_shop') : (int)\Tools::getValue('iShopId');

        $product_category = new \Category((int)$this->data->p->getDefaultCategory(), (int)$id_lang);
        $this->data->step->url = \Context::getContext()->link->getProductLink($this->data->p, null, \Tools::strtolower($product_category->link_rewrite), null, (int)$id_lang, $id_shop, (int) $this->data->c['id_product_attribute'], false, false, false, array(), false);

        //handle the advanced parameters
        // format the current URL with currency or Google campaign parameters
        if (!empty(\FacebookProductAd::$conf['FPA_ADD_CURRENCY'])) {
            $this->data->step->url .= (strpos($this->data->step->url, '?') !== false) ? '&SubmitCurrency=1&id_currency=' . (int)$this->data->currencyId : '?SubmitCurrency=1&id_currency=' . (int)$this->data->currencyId;
        }
        if (!empty(\FacebookProductAd::$conf['FPA_UTM_CAMPAIGN'])) {
            $this->data->step->url .= (strpos($this->data->step->url, '?') !== false) ? '&utm_campaign=' . \FacebookProductAd::$conf['FPA_UTM_CAMPAIGN'] : '?utm_campaign=' . \FacebookProductAd::$conf['FPA_UTM_CAMPAIGN'];
        }
        if (!empty(\FacebookProductAd::$conf['FPA_UTM_SOURCE'])) {
            $this->data->step->url .= (strpos($this->data->step->url, '?') !== false) ? '&utm_source=' . \FacebookProductAd::$conf['FPA_UTM_SOURCE'] : '?utm_source=' . \FacebookProductAd::$conf['FPA_UTM_SOURCE'];
        }
        if (!empty(\FacebookProductAd::$conf['FPA_UTM_MEDIUM'])) {
            $this->data->step->url .= (strpos($this->data->step->url, '?') !== false) ? '&utm_medium=' . \FacebookProductAd::$conf['FPA_UTM_MEDIUM'] : '?utm_medium=' . \FacebookProductAd::$conf['FPA_UTM_MEDIUM'];
        }

        // get weight
        $this->data->step->weight = (float) $this->data->p->weight + (float) $this->data->c['weight'];

        // Use tax according to the option
        $bUseTax = !empty(\FacebookProductAd::$conf['FPA_PROD_PRICE_TAX']) ? true : false;

        // handle different prices and shipping fees
        $this->data->step->price_default_currency_no_tax = \Tools::convertPrice(\Product::getPriceStatic((int) $this->data->p->id, false, (int) $this->data->c['id_product_attribute']), $this->data->currency, false);

        // Exclude based on min price
        if (
            !empty(\FacebookProductAd::$conf['FPA_MIN_PRICE']) && ((float) $this->data->step->price_default_currency_no_tax < (float) \FacebookProductAd::$conf['FPA_MIN_PRICE'])
        ) {
            moduleReporting::create()->set('_no_export_min_price', array('productId' => $this->data->step->id_reporting));
            return false;
        }

        $this->data->step->price_raw = \Product::getPriceStatic((int) $this->data->p->id, $bUseTax, (int) $this->data->c['id_product_attribute']);
        $this->data->step->price_raw_no_discount = \Product::getPriceStatic((int) $this->data->p->id, $bUseTax, (int) $this->data->c['id_product_attribute'], 6, null, false, false);
        $this->data->step->price = number_format(moduleTools::round($this->data->step->price_raw), 2, '.', '') . ' ' . $this->data->currency->iso_code;
        $this->data->step->price_no_discount = number_format(moduleTools::round($this->data->step->price_raw_no_discount), 2, '.', '') . ' ' . $this->data->currency->iso_code;

        // Use case handle pack price accuratlly
        if (\FacebookProductAd::$bAdvancedPack && \AdvancedPack::isValidPack($this->data->p->id)) {
            $oPack = new \AdvancedPack($this->data->p->id);
            $this->data->step->price_raw_no_discount = number_format(\AdvancedPack::getPackPrice($oPack->id, $bUseTax, false), 2, '.', '') . ' ' . $this->data->currency->iso_code;
            $this->data->step->price_raw = number_format(\AdvancedPack::getPackPrice($oPack->id), 2, '.', '') . ' ' . $this->data->currency->iso_code;
            $this->data->step->price_no_discount = number_format(\AdvancedPack::getPackPrice($oPack->id, $bUseTax, false), 2, '.', '') . ' ' . $this->data->currency->iso_code;
            $this->data->step->price = number_format(\AdvancedPack::getPackPrice($oPack->id), 2, '.', '') . ' ' . $this->data->currency->iso_code;
        }

        $this->data->step->availabilty_date = "";

        if ($this->data->c['available_date'] != "0000-00-00") {
            $this->data->step->availabilty_date = $this->data->c['available_date'];
        }
        // shipping fees
        if (!empty(\FacebookProductAd::$conf['FPA_SHIPPING_USE'])) {
            $this->data->step->shipping_fees = number_format((float) $this->getProductShippingFees((float) moduleTools::round($this->data->step->price_raw)), 2, '.', '') . ' ' . $this->data->currency->iso_code;
        }

        // get images
        $this->data->step->images = $this->getImages($this->data->p, $this->data->c['id_product_attribute']);

        // quantity
        // Do not export if the quantity is 0 for the combination and export out of stock setting is not On
        if (
            (int) $this->data->c['combo_quantity'] <= 0
            && (int) \FacebookProductAd::$conf['FPA_EXPORT_OOS'] == 0
        ) {
            moduleReporting::create()->set(
                '_no_export_no_stock',
                array('productId' => $this->data->step->id_reporting)
            );
            return false;
        }
        $this->data->step->quantity = (int) $this->data->c['combo_quantity'];

        // EAN13 or UPC
        $this->data->step->ean13 = $this->data->c['ean13'];
        $this->data->step->upc = !empty($this->data->c['upc']) ? $this->data->c['upc'] : '';

        if (!empty(moduleTools::getGtin(\FacebookProductAd::$conf['FPA_GTIN_PREF'], $this->data->c))) {
            $this->data->step->gtin = moduleTools::getGtin(\FacebookProductAd::$conf['FPA_GTIN_PREF'], $this->data->c);
        } else {
            $this->data->step->gtin = moduleTools::getGtin(\FacebookProductAd::$conf['FPA_GTIN_PREF'], (array)$this->data->p);
        }

        // Exclude without EAN
        if (
            \FacebookProductAd::$conf['FPA_EXC_NO_EAN']
            && (empty($this->data->step->ean13) || \Tools::strlen($this->data->step->ean13) < 10)
            && (empty($this->data->step->upc) || \Tools::strlen($this->data->step->upc) < 10)
        ) {
            moduleReporting::create()->set(
                '_no_export_no_ean_upc',
                array('productId' => $this->data->step->id_reporting)
            );
            return false;
        }

        // supplier reference
        $this->data->step->mpn = $this->getSupplierReference($this->data->p->id, $this->data->p->id_supplier, $this->data->p->supplier_reference, $this->data->p->reference, (int) $this->data->c['id_product_attribute'], $this->data->c['supplier_reference'], $this->data->c['reference']);

        // exclude if mpn is empty
        if (
            !empty(\FacebookProductAd::$conf['FPA_EXC_NO_MREF'])
            && !\FacebookProductAd::$conf['FPA_INC_ID_EXISTS']
            && empty($this->data->step->mpn)
        ) {
            moduleReporting::create()->set(
                '_no_export_no_supplier_ref',
                array('productId' => $this->data->step->id_reporting)
            );
            return false;
        }

        $this->data->step->visibility = $this->data->p->visibility;

        return true;
    }

    /**
     * method format the product name
     *
     * @param int $iAdvancedProdName
     * @param int $iAdvancedProdTitle
     * @param string $sProdName
     * @param string $sCatName
     * @param string $sManufacturerName
     * @param int $iLength
     * @param int $iProdAttrId
     * @return string
     */
    public function formatProductName($iAdvancedProdName, $iAdvancedProdTitle, $sProdName, $sCatName, $sManufacturerName, $iLength, $iProdAttrId = null)
    {
        // get the combination attributes to format the product name
        $sProdName .= moduleTools::getProductCombinationName($iProdAttrId, $this->aParams['iLangId'], $this->aParams['iShopId']);
        // encode
        $sProdName = moduleTools::truncateProductTitle($iAdvancedProdName, $sProdName, $sCatName, $sManufacturerName, $iLength);
        $sProdName = moduleTools::formatProductTitle($sProdName, $iAdvancedProdTitle);

        return $sProdName;
    }

    /**
     * method get images of one product or one combination
     *
     * @param obj $oProduct
     * @param int $iProdAttributeId
     * @return array
     */
    public function getImages(\Product $oProduct, $iProdAttributeId = null)
    {
        // set vars
        $aResultImages = array();
        $iCounter = 1;

        // get images of combination
        $aAttributeImages = $oProduct->getCombinationImages(\FacebookProductAd::$iCurrentLang);

        if (!empty($aAttributeImages[$iProdAttributeId]) && is_array($aAttributeImages[$iProdAttributeId])) {
            $aImage = array('id_image' => $aAttributeImages[$iProdAttributeId][0]['id_image']);
        } else {
            $aImage = \Product::getCover($oProduct->id);
        }

        // Additional images
        unset($aAttributeImages['id_image']);

        if (!empty($aAttributeImages) && is_array($aAttributeImages)) {
            foreach ($aAttributeImages[$iProdAttributeId] as $sImg) {
                if ($iCounter <= (int)moduleConfiguration::FPA_IMG_LIMIT) {
                    $aResultImages[] = array('id_image' => $sImg['id_image']);
                    $iCounter++;
                }
            }
        }

        return array('image' => $aImage, 'others' => $aResultImages);
    }

    /**
     * method get supplier reference
     *
     * @param int $iProdId
     * @param int $iSupplierId
     * @param string $sSupplierRef
     * @param string $sProductRef
     * @param int $iProdAttributeId
     * @param string $sCombiSupplierRef
     * @param string $sCombiRef
     * @return string
     */
    public function getSupplierReference($iProdId, $iSupplierId, $sSupplierRef = null, $sProductRef = null, $iProdAttributeId = 0, $sCombiSupplierRef = null, $sCombiRef = null)
    {
        $sReturnRef = '';

        if (empty(\FacebookProductAd::$bCompare1770)) {
            // detect the MPN type
            $sReturnRef = moduleDao::getProductSupplierReference($iProdId, $iSupplierId, $iProdAttributeId);
        } else {
            $oCombination = new \Combination($iProdAttributeId);

            if (!empty($oCombination->mpn)) {
                $sReturnRef = $oCombination->mpn;
            } else {
                $oProduct = new \Product($iProdId);
                $sReturnRef = $oProduct->mpn;
            }
        }

        return $sReturnRef;
    }
}
