<?php
/**
 * Dynamic Ads + Pixel
 *
 * @author    businesstech.fr <modules@businesstech.fr> - https://www.businesstech.fr/
 * @copyright Business Tech - https://www.businesstech.fr/
 * @license   see file: LICENSE.txt
 *
 *           ____    _______
 *          |  _ \  |__   __|
 *          | |_) |    | |
 *          |  _ <     | |
 *          | |_) |    | |
 *          |____/     |_|
 */

namespace FacebookProductAd\ModuleLib;

use FacebookProductAd\Common\dirReader;
use FacebookProductAd\Common\fileClass;
use \FacebookProductAd\Configuration\moduleConfiguration;
use FacebookProductAd\Models\Feeds;
use FacebookProductAd\Install\installController;

class moduleUpdate
{
    /**
     * @var $aErrors : store errors
     */
    protected $aErrors = array();

    /**
     * Magic Method __construct
     */
    public function __construct()
    {
    }

    /**
     * run() method execute required function
     *
     * @param string $sType
     * @param array $aParam
     */
    public function run($sType, $aParam = null)
    {

        // get type
        $sType = empty($sType) ? 'tables' : $sType;

        switch ($sType) {
            case 'tables': // use case - update tables
            case 'fields': // use case - update fields
            case 'hooks': // use case - update hooks
            case 'templates': // use case - update templates
            case 'moduleAdminTab': // use case - update old module admin tab version
            case 'xmlFiles': // use case - initialize XML files
            case 'feedsDatabaseMigration': // use case - handle the database migration for the feeds
            case 'configuration': // use case - update configuration
                // execute match function
                call_user_func(array($this, 'update' . ucfirst($sType)), $aParam);
                break;
            default:
                break;
        }
    }


    /**
     * _updateTables() method update tables if required
     *
     * @param array $aParam
     */
    private function updateTables(array $aParam = null)
    {
        // set transaction
        if (\Tools::getValue('controller') == 'AdminModules') {
            \Db::getInstance()->Execute('BEGIN');

            if (!empty(moduleConfiguration::getSqlUpdateData()['table'])) {
                $iCount = 1;
                // loop on each elt to update SQL
                foreach (moduleConfiguration::getSqlUpdateData()['table'] as $sTable => $sSqlFile) {
                    // execute query
                    $bResult = \Db::getInstance()->ExecuteS('SHOW TABLES LIKE "' . _DB_PREFIX_ . strtolower(moduleConfiguration::FPA_MODULE_NAME) . '_' . \bqSQL($sTable) . '"');

                    // if empty - update
                    if (empty($bResult)) {

                        // use case - KO update
                        if (!installController::run('install', 'sql', moduleConfiguration::FPA_PATH_SQL . $sSqlFile)) {
                            $this->aErrors[] = array(
                                'msg' => \FacebookProductAd::$oModule->l('There is an error around the SQL table update', 'moduleUpdate'),
                                'code' => intval(190 + $iCount),
                                'file' => $sSqlFile,
                                'context' => \FacebookProductAd::$oModule->l('Issue around table update for: ', 'moduleUpdate') . $sTable
                            );
                            ++$iCount;
                        }
                    }
                }
            }

            if (empty($this->aErrors)) {
                \Db::getInstance()->Execute('COMMIT');
            } else {
                \Db::getInstance()->Execute('ROLLBACK');
            }
        }
    }


    /**
     * _updateFields() method update fields if required
     *
     * @param array $aParam
     */
    private function updateFields(array $aParam = null)
    {
        // set transaction
        \Db::getInstance()->Execute('BEGIN');

        if (!empty(moduleConfiguration::getSqlUpdateData()['field'])) {
            $iCount = 1;
            // loop on each elt to update SQL
            foreach (moduleConfiguration::getSqlUpdateData()['field'] as $sFieldName => $aOption) {
                // execute query
                $bResult = \Db::getInstance()->ExecuteS('SHOW COLUMNS FROM ' . _DB_PREFIX_ . strtolower(moduleConfiguration::FPA_MODULE_NAME) . '_' . \bqSQL($aOption['table']) . ' LIKE "' . \pSQL($sFieldName) . '"');

                // if empty - update
                if (empty($bResult)) {

                    // use case - KO update
                    if (!installController::run('install', 'sql', moduleConfiguration::FPA_PATH_SQL . $aOption['file'])) {
                        $aErrors[] = array(
                            'field' => $sFieldName,
                            'linked' => $aOption['table'],
                            'file' => $aOption['file']
                        );
                        $this->aErrors[] = array(
                            'msg' => \FacebookProductAd::$oModule->l('There is an error around the SQL field update', 'moduleUpdate'),
                            'code' => intval(180 + $iCount),
                            'file' => $aOption['file'],
                            'context' => \FacebookProductAd::$oModule->l('Issue around field update for: ', 'moduleUpdate') . $sFieldName
                        );
                        ++$iCount;
                    }
                }
            }
        }

        if (empty($this->aErrors)) {
            \Db::getInstance()->Execute('COMMIT');
        } else {
            \Db::getInstance()->Execute('ROLLBACK');
        }
    }

    /**
     * _updateHooks() method update hooks if required
     *
     * @param array $aParam
     */
    private function updateHooks(array $aParam = null)
    {
        // use case - hook register ko
        if (!installController::run('install', 'config', array('bHookOnly' => true))) {
            $this->aErrors[] = array(
                'msg' => \FacebookProductAd::$oModule->l('There is an error around the hooks update', 'moduleUpdate'),
                'code' => 170,
                'file' => \FacebookProductAd::$oModule->l('There is an error around the hooks update', 'moduleUpdate'),
                'context' => \FacebookProductAd::$oModule->l('Issue around hook update', 'moduleUpdate')
            );
        }
    }

    /**
     * _updateTemplates() method update templates if required
     *
     * @param array $aParam
     */
    private function updateTemplates(array $aParam = null)
    {
        // get templates files
        $aTplFiles = dirReader::create()->run(array(
            'path' => moduleConfiguration::FPA_PATH_TPL,
            'recursive' => true,
            'extension' => 'tpl',
            'subpath' => true
        ));

        if (!empty($aTplFiles)) {

            $smarty = \Context::getContext()->smarty;

            if (method_exists($smarty, 'clearCompiledTemplate')) {
                $smarty->clearCompiledTemplate();
            } elseif (method_exists($smarty, 'clear_compiled_tpl')) {
                foreach ($aTplFiles as $aFile) {
                    $smarty->clear_compiled_tpl($aFile['filename']);
                }
            }
        }
    }


    /**
     * _updateModuleAdminTab() method update module admin tab in case of an update
     *
     * @param array $aParam
     */
    private function updateModuleAdminTab(array $aParam = null)
    {
        foreach (moduleConfiguration::FPA_TABS as $sModuleTabName => $aTab) {
            if (isset($aTab['oldName'])) {
                if (\Tab::getIdFromClassName($aTab['oldName']) != false) {

                    // use case - if uninstall succeeded
                    if (installController::run('uninstall', 'tab', array('name' => $aTab['oldName']))) {
                        // install new admin tab
                        installController::run('install', 'tab', array('name' => $sModuleTabName));
                    }
                }
            }
        }
    }

    /**
     * _updateXmlFiles() method initialize XML files
     *
     * @param array $aParam
     */
    private function updateXmlFiles(array $aParam = null)
    {
        if (
            !empty($aParam['aAvailableData'])
            && is_array($aParam['aAvailableData'])
        ) {
            $iCount = 1;
            foreach ($aParam['aAvailableData'] as $aData) {
                // check if file exist
                $sFileSuffix = moduleTools::buildFileSuffix($aData['langIso'], $aData['countryIso']);
                $sFilePath = \FacebookProductAd::$sFilePrefix . '.' . $sFileSuffix . '.xml';

                if (!is_file(moduleConfiguration::FPA_SHOP_PATH_ROOT . $sFilePath)) {
                    try {
                        fileClass::create()->write(moduleConfiguration::FPA_SHOP_PATH_ROOT . $sFilePath, '');

                        // test if file exists
                        $bFileExists = is_file(moduleConfiguration::FPA_SHOP_PATH_ROOT . $sFilePath);
                        //					    $bFileExists = false;
                    } catch (\Exception $e) {
                        $bFileExists = false;
                    }

                    if (!$bFileExists) {
                        $aError = array(
                            'msg' => \FacebookProductAd::$oModule->l('There is an error around the creation of the data feed XML file in the shop\'s root directory', 'moduleUpdate'),
                            'code' => intval(160 + $iCount),
                            'file' => moduleConfiguration::FPA_SHOP_PATH_ROOT . $sFilePath,
                            'context' => \FacebookProductAd::$oModule->l('Issue around the xml files which have to be generated in the shop\'s root directory', 'moduleUpdate'),
                            'howTo' => \FacebookProductAd::$oModule->l('Please follow our FAQ about problems when creating XML files at the root of your shop', 'moduleUpdate') . '&nbsp;=>&nbsp;<i class="icon-question-sign"></i>&nbsp;<a href="http://faq.businesstech.fr/faq/21" target="_blank">FAQ</a>'
                        );
                        $this->aErrors[] = $aError;
                        $iCount++;
                    }
                }
            }
        }
    }

    /**
     * _updateConfiguration() method update specific configuration options
     *
     * @param string $sType
     */
    private function updateConfiguration($sType)
    {
        switch ($sType) {
            case 'languages':
                $aHomeCat = \Configuration::get('FPA_HOME_CAT');
                if (empty($aHomeCat)) {
                    $aHomeCat = array();
                    foreach (\FacebookProductAd::$aAvailableLanguages as $aLanguage) {
                        $aHomeCat[$aLanguage['id_lang']] = !empty(moduleConfiguration::FPA_HOME_CAT_NAME[$aLanguage['iso_code']]) ? moduleConfiguration::FPA_HOME_CAT_NAME[$aLanguage['iso_code']] : '';
                    }
                    // update
                    \Configuration::updateValue('FPA_HOME_CAT', serialize($aHomeCat));
                    unset($aHomeCat);
                } elseif (is_array(\FacebookProductAd::$conf['FPA_HOME_CAT'])) {
                    // update
                    \Configuration::updateValue(
                        'FPA_HOME_CAT',
                        serialize(\FacebookProductAd::$conf['FPA_HOME_CAT'])
                    );
                }
                break;
            case 'color':
                if (!empty(\FacebookProductAd::$conf['FPA_COLOR_OPT'])) {
                    if (is_numeric(\FacebookProductAd::$conf['FPA_COLOR_OPT'])) {
                        \FacebookProductAd::$conf['FPA_COLOR_OPT'] = array(\FacebookProductAd::$conf['FPA_COLOR_OPT']);

                        $aAttributeIds = array();
                        foreach (\FacebookProductAd::$conf['FPA_COLOR_OPT'] as $iAttributeId) {
                            $aAttributeIds['attribute'][] = $iAttributeId;
                        }
                        \Configuration::updateValue('FPA_COLOR_OPT', serialize($aAttributeIds));
                    }
                }
                break;
            default:
                break;
        }
    }

    /**
     * updateFeedsDatabaseMigration() method made the migration in database for the data feeds to replace global used before the version 1.4.0
     *
     */
    private function updateFeedsDatabaseMigration()
    {
        $hasData = Feeds::hasSavedData(\FacebookProductAd::$iShopId);

        if (!empty(moduleConfiguration::FPA_AVAILABLE_COUNTRIES) && empty($hasData)) {
            foreach (moduleConfiguration::FPA_AVAILABLE_COUNTRIES as $lang_code => $data) {
                if (is_array($data)) {
                    foreach ($data as $country_code => $data_entry) {

                        if (is_array($data_entry) && isset($data_entry['currency'])) {
                            foreach ($data_entry['currency'] as $currency) {
                                $feed = new Feeds();
                                $feed->iso_lang = $lang_code;
                                $feed->iso_country = $country_code;
                                $feed->iso_currency = $currency;
                                $feed->taxonomy = $data_entry['taxonomy'];
                                $feed->id_shop = \FacebookProductAd::$iShopId;
                                $feed->add();
                            }
                        }
                    }
                }
            }

            return \Tools::redirectAdmin(\Context::getContext()->link->getAdminLink('AdminModules') . '&configure=facebookproductad');
        }
    }


    /**
     * getErrors() method returns errors
     *
     * @return array
     */
    public function getErrors()
    {
        return (empty($this->aErrors) ? false : $this->aErrors);
    }

    /**
     * create() method manages singleton
     *
     * @param
     * @return array
     */
    public static function create()
    {
        static $oModuleUpdate;

        if (null === $oModuleUpdate) {
            $oModuleUpdate = new moduleUpdate();
        }
        return $oModuleUpdate;
    }
}
