<?php
/*
 * @package   gmcrop
 * @author    Dariusz Tryba (contact@greenmousestudio.com)
 * @copyright Copyright (c) Green Mouse Studio (http://www.greenmousestudio.com)
 * @license   http://greenmousestudio.com/paid-license.txt
 */

class ImageManager extends ImageManagerCore
{

    public static function resize($sourceFile, $destinationFile, $destinationWidth = null, $destinationHeight = null,
                                  $fileType = 'jpg', $forceType = false, &$error = 0, &$targetWidth = null,
                                  &$targetHeight = null, $quality = 5, &$sourceWidth = null, &$sourceHeight = null)
    {
        clearstatcache(true, $sourceFile);

        if (!file_exists($sourceFile) || !filesize($sourceFile)) {
            return !($error = self::ERROR_FILE_NOT_EXIST);
        }

        list($tmpWidth, $tmpHeight, $type) = getimagesize($sourceFile);

        $rotate = 0;
        if (function_exists('exif_read_data') && function_exists('mb_strtolower')) {
            $exif = @exif_read_data($sourceFile);

            if ($exif && isset($exif['Orientation'])) {
                switch ($exif['Orientation']) {
                    case 3:
                        $sourceWidth = $tmpWidth;
                        $sourceHeight = $tmpHeight;
                        $rotate = 180;
                        break;

                    case 6:
                        $sourceWidth = $tmpHeight;
                        $sourceHeight = $tmpWidth;
                        $rotate = -90;
                        break;

                    case 8:
                        $sourceWidth = $tmpHeight;
                        $sourceHeight = $tmpWidth;
                        $rotate = 90;
                        break;

                    default:
                        $sourceWidth = $tmpWidth;
                        $sourceHeight = $tmpHeight;
                }
            } else {
                $sourceWidth = $tmpWidth;
                $sourceHeight = $tmpHeight;
            }
        } else {
            $sourceWidth = $tmpWidth;
            $sourceHeight = $tmpHeight;
        }

        //create image first
        $srcImage = ImageManager::create($type, $sourceFile);

        if (Module::isEnabled('gmcrop')) {
            if (function_exists('imagecropauto')) {
                //autocrop
                if (Configuration::get('GMCROP_CROPBG')) {
                    //attempt to crop the transparent background
					$cropped = imagecropauto($srcImage, IMG_CROP_THRESHOLD, 0.25, 16777215);
					if ($cropped !== false) {
						imagedestroy($srcImage);
						$srcImage = $cropped;
						//get new size after autocrop
						$sourceWidth = imagesx($srcImage);
						$sourceHeight = imagesy($srcImage);
					}
                }
            }
            //keep proportions mode
            $keepProportions = false;
            $fittedImageTypes = explode(',', Configuration::get('GMCROP_FITTEDTYPES'));
            if (is_array($fittedImageTypes) && (count($fittedImageTypes) > 0)) {
                foreach ($fittedImageTypes as $fittedImageType) {
                    if (strlen($fittedImageType) > 1) {
                        if (strpos($destinationFile, $fittedImageType) !== false) {
                            $keepProportions = true;
                        }
                    }
                }
            }
            if ($keepProportions) {
                $widthRatio = $destinationWidth / $sourceWidth;
                $heightRatio = $destinationHeight / $sourceHeight;
                if (($widthRatio < 1) || ($heightRatio < 1)) {
                    if ($widthRatio < $heightRatio) {
                        $destinationWidth = $sourceWidth * $widthRatio;
                        $destinationHeight = $sourceHeight * $widthRatio;
                    } else {
                        $destinationWidth = $sourceWidth * $heightRatio;
                        $destinationHeight = $sourceHeight * $heightRatio;
                    }
                } else {
                    $destinationWidth = $sourceWidth;
                    $destinationHeight = $sourceHeight;
                }
            }
        }
        // If PS_IMAGE_QUALITY is activated, the generated image will be a PNG with .jpg as a file extension.
        // This allow for higher quality and for transparency. JPG source files will also benefit from a higher quality
        // because JPG reencoding by GD, even with max quality setting, degrades the image.
        if (Configuration::get('PS_IMAGE_QUALITY') == 'png_all' || (Configuration::get('PS_IMAGE_QUALITY') == 'png' && $type
            == IMAGETYPE_PNG) && !$forceType) {
            $fileType = 'png';
        }

        if (!$sourceWidth) {
            return !($error = self::ERROR_FILE_WIDTH);
        }
        if (!$destinationWidth) {
            $destinationWidth = $sourceWidth;
        }
        if (!$destinationHeight) {
            $destinationHeight = $sourceHeight;
        }

        $widthDiff = $destinationWidth / $sourceWidth;
        $heightDiff = $destinationHeight / $sourceHeight;

        $rgb = ImageManager::hex2rgb(Configuration::get('GMCROP_BGCOLOR'));
        $red = $rgb[0];
        $green = $rgb[1];
        $blue = $rgb[2];

        $crop = false;
        if (Module::isEnabled('gmcrop')) {
            $croppedImageTypes = explode(',', Configuration::get('GMCROP_CROPPEDTYPES'));
            if (is_array($croppedImageTypes) && (count($croppedImageTypes) > 0)) {
                foreach ($croppedImageTypes as $croppedImageType) {
                    if (strlen($croppedImageType) > 1) {
                        if (strpos($destinationFile, $croppedImageType) !== false) {
                            $crop = true;
                        }
                    }
                }
            }
        } else {
            $red = $green = $blue = 255;
        }
        if (!$crop) {
            $psImageGenerationMethod = Configuration::get('PS_IMAGE_GENERATION_METHOD');
            if ($widthDiff > 1 && $heightDiff > 1) {
                $nextWidth = $sourceWidth;
                $nextHeight = $sourceHeight;
            } else {
                if ($psImageGenerationMethod == 2 || (!$psImageGenerationMethod && $widthDiff > $heightDiff)) {
                    $nextHeight = $destinationHeight;
                    $nextWidth = round(($sourceWidth * $nextHeight) / $sourceHeight);
                    $destinationWidth = (int) (!$psImageGenerationMethod ? $destinationWidth : $nextWidth);
                } else {
                    $nextWidth = $destinationWidth;
                    $nextHeight = round($sourceHeight * $destinationWidth / $sourceWidth);
                    $destinationHeight = (int) (!$psImageGenerationMethod ? $destinationHeight : $nextHeight);
                }
            }

            if (!ImageManager::checkImageMemoryLimit($sourceFile)) {
                return !($error = self::ERROR_MEMORY_LIMIT);
            }

            $targetWidth = $destinationWidth;
            $targetHeight = $destinationHeight;

            $destImage = imagecreatetruecolor($destinationWidth, $destinationHeight);

            // If image is a PNG and the output is PNG, fill with transparency. Else fill with white background.
            if ($fileType == 'png' && $type == IMAGETYPE_PNG) {
                imagealphablending($destImage, false);
                imagesavealpha($destImage, true);
                $transparent = imagecolorallocatealpha($destImage, $red, $green, $blue, 127);
                imagefilledrectangle($destImage, 0, 0, $destinationWidth, $destinationHeight, $transparent);
            } else {
                $white = imagecolorallocate($destImage, $red, $green, $blue);
                imagefilledrectangle($destImage, 0, 0, $destinationWidth, $destinationHeight, $white);
            }

            //$src_image = ImageManager::create($type, $src_file);
            if ($rotate) {
                $srcImage = imagerotate($srcImage, $rotate, 0);
            }

            if ($destinationWidth >= $sourceWidth && $destinationHeight >= $sourceHeight) {
                imagecopyresized($destImage, $srcImage, (int) (($destinationWidth - $nextWidth) / 2),
                    (int) (($destinationHeight - $nextHeight) / 2), 0, 0, $nextWidth, $nextHeight, $sourceWidth,
                    $sourceHeight);
            } else {
                ImageManager::imagecopyresampled($destImage, $srcImage, (int) (($destinationWidth - $nextWidth) / 2),
                    (int) (($destinationHeight - $nextHeight) / 2), 0, 0, $nextWidth, $nextHeight, $sourceWidth,
                    $sourceHeight, $quality);
            }
            $writeFile = ImageManager::write($fileType, $destImage, $destinationFile);
            @imagedestroy($srcImage);
            return $writeFile;
        } else {
            $psImageGenerationMethod = Configuration::get('PS_IMAGE_GENERATION_METHOD');
            if ($widthDiff > 1 && $heightDiff > 1) {
                $nextWidth = $sourceWidth;
                $nextHeight = $sourceHeight;
            } else {
                if ($psImageGenerationMethod == 2 || (!$psImageGenerationMethod && $widthDiff < $heightDiff)) {
                    $nextHeight = $destinationHeight;
                    $nextWidth = round(($sourceWidth * $nextHeight) / $sourceHeight);
                    $destinationWidth = (int) (!$psImageGenerationMethod ? $destinationWidth : $nextWidth);
                } else {
                    $nextWidth = $destinationWidth;
                    $nextHeight = round($sourceHeight * $destinationWidth / $sourceWidth);
                    $destinationHeight = (int) (!$psImageGenerationMethod ? $destinationHeight : $nextHeight);
                }
            }

            if (!ImageManager::checkImageMemoryLimit($sourceFile)) {
                return !($error = self::ERROR_MEMORY_LIMIT);
            }

            $targetWidth = $destinationWidth;
            $targetHeight = $destinationHeight;

            $destImage = imagecreatetruecolor($destinationWidth, $destinationHeight);

            // If image is a PNG and the output is PNG, fill with transparency. Else fill with white background.
            if ($fileType == 'png' && $type == IMAGETYPE_PNG) {
                imagealphablending($destImage, false);
                imagesavealpha($destImage, true);
                $transparent = imagecolorallocatealpha($destImage, $red, $green, $blue, 127);
                imagefilledrectangle($destImage, 0, 0, $destinationWidth, $destinationHeight, $transparent);
            } else {
                $white = imagecolorallocate($destImage, $red, $green, $blue);
                imagefilledrectangle($destImage, 0, 0, $destinationWidth, $destinationHeight, $white);
            }

            //$src_image = ImageManager::create($type, $src_file);
            if ($rotate) {
                $srcImage = imagerotate($srcImage, $rotate, 0);
            }

            $originConfig = Configuration::get('GMCROP_ORIGIN');
            switch ($originConfig) {
                case 0:
                    $originX = 0;
                    $originY = 0;
                    break;
                case 1:
                    $originX = (int) (($destinationWidth - $nextWidth) / 2);
                    $originY = 0;
                    break;
                case 2:
                    $originX = (int) ($destinationWidth - $nextWidth);
                    $originY = 0;
                    break;
                case 3:
                    $originX = 0;
                    $originY = (int) (($destinationHeight - $nextHeight) / 2);
                    break;
                case 4:
                    $originX = (int) (($destinationWidth - $nextWidth) / 2);
                    $originY = (int) (($destinationHeight - $nextHeight) / 2);
                    break;
                case 5:
                    $originX = (int) ($destinationWidth - $nextWidth);
                    $originY = (int) (($destinationHeight - $nextHeight) / 2);
                    break;
                case 6:
                    $originX = 0;
                    $originY = (int) (($destinationHeight - $nextHeight));
                    break;
                case 7:
                    $originX = (int) (($destinationWidth - $nextWidth) / 2);
                    $originY = (int) (($destinationHeight - $nextHeight));
                    break;
                case 8:
                    $originX = (int) ($destinationWidth - $nextWidth);
                    $originY = (int) (($destinationHeight - $nextHeight));
                    break;
                default:
                    $originX = (int) (($destinationWidth - $nextWidth) / 2);
                    $originY = (int) (($destinationHeight - $nextHeight) / 2);
                    break;
            }
            if ($destinationWidth >= $sourceWidth && $destinationHeight >= $sourceHeight) {
                imagecopyresized($destImage, $srcImage, $originX, $originY, 0, 0, $nextWidth, $nextHeight, $sourceWidth,
                    $sourceHeight);
            } else {
                ImageManager::imagecopyresampled($destImage, $srcImage, $originX, $originY, 0, 0, $nextWidth,
                    $nextHeight, $sourceWidth, $sourceHeight, $quality);
            }
            $writeFile = ImageManager::write($fileType, $destImage, $destinationFile);
            Hook::exec('actionOnImageResizeAfter', array('dst_file' => $destinationFile, 'file_type' => $fileType));
            @imagedestroy($srcImage);

            file_put_contents(
                dirname($destinationFile).DIRECTORY_SEPARATOR.'fileType', $fileType
            );
            return $writeFile;
        }
    }

    public static function hex2rgb($hex)
    {
        $hex = str_replace("#", "", $hex);
        if (strlen($hex) == 3) {
            $r = hexdec(substr($hex, 0, 1).substr($hex, 0, 1));
            $g = hexdec(substr($hex, 1, 1).substr($hex, 1, 1));
            $b = hexdec(substr($hex, 2, 1).substr($hex, 2, 1));
        } else {
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
        }
        $rgb = array($r, $g, $b);
        return $rgb; // returns an array with the rgb values
    }
}
