<?php
/**
 * 2007-2020 PrestaShop and Contributors
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License 3.0 (AFL-3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş. <devops@paytr.com>
 * @copyright PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş.
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 * International Registered Trademark & Property of PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş.
 */

require_once(dirname(__FILE__) . '/PaytrTransaction.php');
require_once(dirname(__FILE__) . '/PaytrTransactionForEFTByNotification.php');

class PaytrNotification
{
    public $context;
    public $module;

    public function iframeCallback($params, $uniq_separator)
    {
        // Check Hash
        $this->checkHash($params, 'iframe');

        // Get ID Cart
        $id_cart = explode($uniq_separator, $params['merchant_oid'], 2);

        // Get PayTR Transaction
        $paytr_transaction = PaytrTransaction::getTransactionByIdOidCartByOid($id_cart[1], $params['merchant_oid']);

        if (!$paytr_transaction) {
            die('OK');
        }

        // If transaction is already completed
        if ($paytr_transaction['is_complete'] && $paytr_transaction['status'] == 'success' || $paytr_transaction['status'] == 'failed') {
            echo 'OK';
            exit;
        }

        // Get Configurations
        $conf_pending = Configuration::get('PAYTR_PENDING');
        $conf_canceled = Configuration::get('PAYTR_CANCELED');
        $conf_completed = Configuration::get('PAYTR_COMPLETED');
        $conf_invoice = Configuration::get('PAYTR_INVOICE');

        // Amounts
        // Total Paid (Included installment difference if exist)
        // Payment Amount (Cart Total)
        $total_amount = round($params['total_amount'] / 100, 2);
        $payment_amount = round($params['payment_amount'] / 100, 2);

        $val = array(
            'status' => $params['status'],
            'total_amount' => $total_amount,
            'payment_amount' => $payment_amount,
            'conf_pending' => $conf_pending,
            'conf_completed' => $conf_completed,
            'conf_canceled' => $conf_canceled,
            'conf_invoice' => $conf_invoice,
            'merchant_oid' => $params['merchant_oid'],
            'payment_type' => $params['payment_type']
        );

        if (array_key_exists('installment_count', $params)) {
            $val['installment_count'] = $params['installment_count'];
        }

        if ($params['status'] == 'failed') {
            $val['failed_reason_code'] = $params['failed_reason_code'];
            $val['failed_reason_msg'] = $params['failed_reason_msg'];
        }

        // Check order if already placed before.
        // Some Prestashop users' systems creating orders when payment page.
        $ps_order_id = $this->checkOrder((int)$id_cart[1]);

        if ($ps_order_id) {
            $order = new Order($ps_order_id);

            // check order status
            if ($order->current_state != $conf_pending) {
                // State 12 = Out of Stock (Pre-Order Not Paid)
                if ($params['status'] == 'success' && $order->current_state == 12) {
                    $order->setCurrentState(9); // Out of Stock (Pre-Order Paid)
                    PaytrTransaction::updateTransactionIsComplete($paytr_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'completed', $total_amount, 0);
                } else {
                    PaytrTransaction::updateTransactionIsComplete($paytr_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'done', $total_amount, 0);
                }

                echo 'OK';
                exit;
            }

            $this->updateStatus($val, $order, $id_cart[1], 0, 'iframe');
        }

        if ($paytr_transaction['is_order'] && !$paytr_transaction['is_complete']) {
            // Get Order By id_oid_cart
            $order_id = $this->checkOrder((int)$id_cart[1]);

            if (!$order_id) {
                die('OK');
            }

            $order = new Order($order_id);

            // check order status
            if ($order->current_state != $conf_pending) {
                // State 12 = Out of Stock (Pre-Order Not Paid)
                if ($params['status'] == 'success' && $order->current_state == 12) {
                    $order->setCurrentState(9); // Out of Stock (Pre-Order Paid)
                    PaytrTransaction::updateTransactionIsComplete($paytr_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'completed', $total_amount, 0);
                } else {
                    PaytrTransaction::updateTransactionIsComplete($paytr_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'done', $total_amount, 0);
                }

                echo 'OK';
                exit;
            }

            $this->updateStatus($val, $order, $id_cart[1], 0, 'iframe');
        } elseif (!$paytr_transaction['is_order']) {
            if ($params['status'] == 'failed' && $params['failed_reason_code'] == 6) {
                echo 'OK';
                exit;
            }

            // Get Cart
            $cart = new Cart($paytr_transaction['id_oid_cart']);

            $currency = new Currency($cart->id_currency);
            $total = (float)$cart->getOrderTotal(true, Cart::BOTH);
            $customer = new Customer($cart->id_customer);

            // Create New Order
            $this->module->validateOrder($cart->id, $conf_pending, $total, $this->module->displayName, null, null, (int)$currency->id, false, $customer->secure_key);

            $order = new Order($this->module->currentOrder);

            $this->updateStatus($val, $order, $id_cart[1], 1, 'iframe');
        }
    }

    public function eftCallback($params, $uniq_separator)
    {
        // Check Hash
        $this->checkHash($params, 'eft');

        // Get ID Cart
        $id_cart = explode($uniq_separator, $params['merchant_oid'], 2);

        // Get PayTR Transaction
        $paytr_eft_transaction = PaytrTransactionForEFTByNotification::getTransactionByIdOidCartByOid($id_cart[1], $params['merchant_oid']);

        if (!$paytr_eft_transaction) {
            die('OK.');
        }

        // If transaction is already completed
        if ($paytr_eft_transaction['is_complete'] && $paytr_eft_transaction['status'] == 'success' || $paytr_eft_transaction['status'] == 'failed') {
            echo 'OK';
            exit;
        }

        // Get Configurations
        $conf_pending = Configuration::get('PAYTR_EFT_PENDING');
        $conf_canceled = Configuration::get('PAYTR_EFT_CANCELED');
        $conf_completed = Configuration::get('PAYTR_EFT_COMPLETED');

        // Amounts
        // Total Paid (Included installment difference if exist)
        // Payment Amount (Cart Total)
        $total_amount = round($params['total_amount'] / 100, 2);
        $payment_amount = round($params['payment_amount'] / 100, 2);

        $val = array(
            'status' => $params['status'],
            'total_amount' => $total_amount,
            'payment_amount' => $payment_amount,
            'conf_pending' => $conf_pending,
            'conf_completed' => $conf_completed,
            'conf_canceled' => $conf_canceled,
            'merchant_oid' => $params['merchant_oid'],
            'payment_type' => $params['payment_type']
        );

        if (array_key_exists('installment_count', $params)) {
            $val['installment_count'] = $params['installment_count'];
        }

        if ($params['status'] == 'failed') {
            $val['failed_reason_code'] = $params['failed_reason_code'];
            $val['failed_reason_msg'] = $params['failed_reason_msg'];
        }

        // Check order if already placed before.
        // Some Prestashop users' systems creating orders when payment page.
        $ps_order_id = $this->checkOrder((int)$id_cart[1]);

        if ($ps_order_id) {
            $order = new Order($ps_order_id);

            // check order status
            if ($order->current_state != $conf_pending) {
                // State 12 = Out of Stock (Pre-Order Not Paid)
                if ($params['status'] == 'success' && $order->current_state == 12) {
                    $order->setCurrentState(9); // Out of Stock (Pre-Order Paid)
                    PaytrTransactionForEFTByNotification::updateTransactionIsComplete($paytr_eft_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'completed', $total_amount, 0);
                } else {
                    PaytrTransactionForEFTByNotification::updateTransactionIsComplete($paytr_eft_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'done', $total_amount, 0);
                }

                echo 'OK';
                exit;
            }

            $this->updateStatus($val, $order, $id_cart[1], 0, 'eft');
        }

        if ($paytr_eft_transaction['is_order'] && !$paytr_eft_transaction['is_complete']) {
            // Get Order By id_oid_cart
            if (version_compare(_PS_VERSION_, '1.7.1.0', '>=')) {
                $order_id = Order::getIdByCartId((int)$id_cart[1]);
            } else {
                $order_id = Order::getOrderByCartId((int)$id_cart[1]);
            }

            if (!$order_id) {
                die('OK');
            }

            $order = new Order($order_id);

            // check order status
            if ($order->current_state != $conf_pending) {
                // State 12 = Out of Stock (Pre-Order Not Paid)
                if ($params['status'] == 'success' && $order->current_state == 12) {
                    $order->setCurrentState(9); // Out of Stock (Pre-Order Paid)
                    PaytrTransactionForEFTByNotification::updateTransactionIsComplete($paytr_eft_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'completed', $total_amount, 0);
                } else {
                    PaytrTransactionForEFTByNotification::updateTransactionIsComplete($paytr_eft_transaction['id_oid_cart'], $params['merchant_oid'], 'success', 'done', $total_amount, 0);
                }

                echo 'OK';
                exit;
            }

            $this->updateStatus($val, $order, $id_cart[1], 0, 'eft');
        } elseif (!$paytr_eft_transaction['is_order']) {
            if ($params['status'] == 'failed' && $params['failed_reason_code'] == 6) {
                echo 'OK';
                exit;
            }

            // Get Cart
            $cart = new Cart($paytr_eft_transaction['id_oid_cart']);

            $currency = new Currency($cart->id_currency);
            $total = (float)$cart->getOrderTotal(true, Cart::BOTH);
            $customer = new Customer($cart->id_customer);

            // Create New Order
            $this->module->validateOrder($cart->id, $conf_pending, $total, $this->module->displayName, null, null, (int)$currency->id, false, $customer->secure_key);

            $order = new Order($this->module->currentOrder);

            $this->updateStatus($val, $order, $id_cart[1], 1, 'eft');
        }
    }

    protected function checkOrder($id_cart)
    {
        $order_id = 0;

        if (version_compare(_PS_VERSION_, '1.7.1.0', '>=')) {
            $order_id = Order::getIdByCartId((int)$id_cart);
        } else {
            $order_id = Order::getOrderByCartId((int)$id_cart);
        }

        return $order_id;
    }

    protected function checkHash($params, $api_name)
    {
        $merchant = array();

        if ($api_name == 'iframe') {
            $merchant['merchant_id'] = Configuration::get('PAYTR_MERCHANT_ID');
            $merchant['merchant_key'] = Configuration::get('PAYTR_MERCHANT_KEY');
            $merchant['merchant_salt'] = Configuration::get('PAYTR_MERCHANT_SALT');
        }

        if ($api_name == 'eft') {
            $merchant['merchant_id'] = Configuration::get('PAYTR_EFT_MERCHANT_ID');
            $merchant['merchant_key'] = Configuration::get('PAYTR_EFT_MERCHANT_KEY');
            $merchant['merchant_salt'] = Configuration::get('PAYTR_EFT_MERCHANT_SALT');
        }

        $created_hash = base64_encode(hash_hmac('sha256', $params['merchant_oid'] . $merchant['merchant_salt'] . $params['status'] . $params['total_amount'], $merchant['merchant_key'], true));

        if ($created_hash != $params['hash']) {
            die('PAYTR notification failed: bad hash.');
        }

        return true;
    }

    protected function updateStatus(array $val, $order, $id_cart, $is_notify, $api_name)
    {
        // Update Order Status
        if ($val['status'] == 'success') {
            if ($order->current_state == $val['conf_pending']) {
                $order->setCurrentState($val['conf_completed']);
            }

            if ($order->current_state == 12) {
                $order->setCurrentState(9);
            }

            // Calculate installment difference
            $installment_amount = $val['total_amount'] - $val['payment_amount'];

            // Get Invoice
            $invoice = new OrderInvoice($order->invoice_number);
            $note = '';

            if ($api_name != 'eft') {
                // Update Invoice
                if ($invoice && $val['conf_invoice'] && $installment_amount > 0) {
                    // Currency
                    $currency = new Currency($order->id_currency);

                    if (version_compare(_PS_VERSION_, '1.7.6.0', '>=')) {
                        $note .= "Vade Farkı : " . $this->context->currentLocale->formatPrice($installment_amount, $currency->iso_code) . "\n";
                        $note .= "Vade Farkı Dahil Toplam Tutar : " . $this->context->currentLocale->formatPrice($val['total_amount'], $currency->iso_code);

                        if (array_key_exists('installment_count', $val)) {
                            $note .= "\n" . "Taksit Sayısı : " . ($val['installment_count'] == 1 ? "Tek Çekim" : $val['installment_count']);
                        }
                    } else {
                        $note .= "Vade Farkı : " . Tools::displayPrice($installment_amount, $currency->iso_code) . "\n";
                        $note .= "Vade Farkı Dahil Toplam Tutar : " . Tools::displayPrice($val['total_amount'], $currency->iso_code);

                        if (array_key_exists('installment_count', $val)) {
                            $note .= "\n" . "Taksit Sayısı : " . ($val['installment_count'] == 1 ? "Tek Çekim" : $val['installment_count']);
                        }
                    }

                    PaytrTransaction::updateOrderInvoice($invoice->number, $order->id, $note);
                }
            }

            // Update Transaction
            if ($api_name == 'eft') {
                PaytrTransactionForEFTByNotification::updateTransactionIsComplete($id_cart, $val['merchant_oid'], $val['status'], 'completed', $val['total_amount'], $is_notify);
            } else {
                PaytrTransaction::updateTransactionIsComplete($id_cart, $val['merchant_oid'], $val['status'], 'completed', $val['total_amount'], $is_notify);
            }
        } elseif ($val['status'] == 'failed' && $val['failed_reason_code'] != '6') {
            $order->setCurrentState($val['conf_canceled']);

            // Update Transaction
            $status_message = $val['failed_reason_code'] . " - " . $val['failed_reason_msg'];

            if ($api_name == 'eft') {
                PaytrTransactionForEFTByNotification::updateTransactionIsComplete($id_cart, $val['merchant_oid'], $val['status'], $status_message, 0);
            } else {
                PaytrTransaction::updateTransactionIsComplete($id_cart, $val['merchant_oid'], $val['status'], $status_message, 0);
            }
        }

        echo 'OK';
        exit;
    }
}
