<?php
/**
 * 2007-2020 PrestaShop and Contributors
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License 3.0 (AFL-3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş. <devops@paytr.com>
 * @copyright PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş.
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 * International Registered Trademark & Property of PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş.
 */

if (!defined('_PS_VERSION_')) {
    exit;
}
require_once(dirname(__FILE__) . '/classes/PaytrInstall.php');

class PaytrCheckout extends PaymentModule
{
    private $contentHtml;
    private $postErrors = array();

    public function __construct()
    {
        $this->name = 'paytrcheckout';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'PayTR Ödeme ve Elektronik Para Kuruluşu A.Ş.';
        $this->need_instance = 1;

        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('PAYTR iFrame API');
        $this->description = $this->l('The infrastructure required to receive payments through PrestaShop with your PayTR membership.');

        $this->confirmUninstall = $this->l('Do you want to uninstall PAYTR module?');

        $this->limited_currencies = array('TRY', 'TL', 'EUR', 'USD', 'GBP', 'RUB');

        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);

        if (!count(Currency::checkPaymentCurrencies($this->id))) {
            $this->warning = $this->l('No currency has been set for this module.');
        }
    }

    public function install()
    {
        if (extension_loaded('curl') == false) {
            $this->postErrors[] = $this->l('You have to enable the cURL extension on your server to install this module.');
            return false;
        }

        if (!parent::install() ||
            !$this->registerHook('header') ||
            !$this->registerHook('payment') ||
            !$this->registerHook('displayAdminOrder') ||
            !$this->registerHook('paymentReturn') ||
            !$this->registerHook('paymentOptions')) {
            return false;
        }

        $install = new PaytrInstall();
        $install->createOrderTable();
        $install->createOrderState();
        $install->updateConfigurations();

        $source = dirname(__FILE__) . '/views/img/os.gif';
        $destination_pending = dirname(__FILE__) . '/../../img/os/' . Configuration::get('PAYTR_PENDING') . '.gif';
        copy($source, $destination_pending);

        return true;
    }

    public function uninstall()
    {
        $uninstall = new PaytrInstall();
        $uninstall->deleteConfigurations();

        return parent::uninstall();
    }

    protected function installmentOptions($categoryBased = false)
    {
        $installments = array();

        for ($i = 0; $i <= 12; $i++) {
            if ($i == 0) {
                $installments[] = array('id_option' => 0, 'name' => $this->l('All Installment Options'));
            } elseif ($i == 1) {
                $installments[] = array('id_option' => 1, 'name' => $this->l('One Shot (No Installment)'));
            } else {
                $installments[] = array('id_option' => $i, 'name' => $this->l('Up to') . " " . $i . " " . $this->l('Installment'));
            }
        }

        if ($categoryBased) {
            array_push($installments, array('id_option' => 13, 'name' => $this->l('CATEGORY BASED')));
        }

        return $installments;
    }

    protected function getConfigFormValues()
    {
        $all_inputs = array();
        $category_based = array();
        $languages = Language::getLanguages(false);

        $form_inputs = array(
            'paytr_merchant_id' => Tools::getValue('paytr_merchant_id', Configuration::get('PAYTR_MERCHANT_ID')),
            'paytr_merchant_key' => Tools::getValue('paytr_merchant_key', Configuration::get('PAYTR_MERCHANT_KEY')),
            'paytr_merchant_salt' => Tools::getValue('paytr_merchant_salt', Configuration::get('PAYTR_MERCHANT_SALT')),
            'paytr_logo' => Tools::getValue('paytr_logo', Configuration::get('PAYTR_LOGO')),
            'paytr_completed' => Tools::getValue('paytr_completed', Configuration::get('PAYTR_COMPLETED')),
            'paytr_canceled' => Tools::getValue('paytr_canceled', Configuration::get('PAYTR_CANCELED')),
            'paytr_invoice' => Tools::getValue('paytr_invoice', Configuration::get('PAYTR_INVOICE')),
            'paytr_installment' => Tools::getValue('paytr_installment', Configuration::get('PAYTR_INSTALLMENT')),
            'paytr_refund' => Tools::getValue('paytr_refund', Configuration::get('PAYTR_REFUND'))
        );

        foreach ($languages as $language) {
            $form_inputs['paytr_method_title'][$language['id_lang']] = Tools::getValue('paytr_method_title', Configuration::get('PAYTR_METHOD_TITLE_' . $language['id_lang']));
        }

        if (Configuration::get('PAYTR_INSTALLMENT') == 13) {
            $array_CI = unserialize(Configuration::get('PAYTR_CI'));

            foreach ($array_CI as $key => $val) {
                $category_based['CI[' . $key . ']'] = $val;
            }

            $all_inputs = array_merge($form_inputs, $category_based);
        } else {
            $all_inputs = $form_inputs;
        }

        return $all_inputs;
    }

    protected function displayFormInfo()
    {
        $this->context->smarty->assign(
            [
                'img' => Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/views/img/logo.png')
            ]
        );

        return $this->display(__FILE__, 'views/templates/admin/info.tpl');
    }

    protected function displayForm()
    {
        // Order States
        $orderStates = OrderState::getOrderStates((int)$this->context->language->id);

        // Installment Options
        $installment_options = $this->installmentOptions(true);

        // Status Options
        $status_options = array(
            array('id_option' => 0, 'name' => $this->l('Disabled')),
            array('id_option' => 1, 'name' => $this->l('Enabled'))
        );

        // Form Inputs
        $form_inputs = array(
            array(
                'type' => 'text',
                'label' => $this->l('Merchant Id'),
                'name' => 'paytr_merchant_id',
                'class' => 'md',
                'required' => true
            ),
            array(
                'type' => 'text',
                'label' => $this->l('Merchant Key'),
                'name' => 'paytr_merchant_key',
                'class' => 'md',
                'required' => true
            ),
            array(
                'type' => 'text',
                'label' => $this->l('Merchant Salt'),
                'name' => 'paytr_merchant_salt',
                'class' => 'md',
                'required' => true
            ),
            array(
                'type' => 'select',
                'label' => $this->l('Logo Show/Hide'),
                'name' => 'paytr_logo',
                'options' => array(
                    'query' => $status_options,
                    'id' => 'id_option',
                    'name' => 'name',
                )
            ),
            array(
                'type' => 'text',
                'label' => $this->l('Payment Method Title'),
                'name' => 'paytr_method_title',
                'class' => 'md',
                'lang' => true
            ),
            array(
                'type' => 'select',
                'label' => $this->l('When Payment Successful'),
                'name' => 'paytr_completed',
                'options' => array(
                    'query' => $orderStates,
                    'id' => 'id_order_state',
                    'name' => 'name'
                )
            ),
            array(
                'type' => 'select',
                'label' => $this->l('When Payment Failed'),
                'name' => 'paytr_canceled',
                'options' => array(
                    'query' => $orderStates,
                    'id' => 'id_order_state',
                    'name' => 'name'
                )
            ),
            array(
                'type' => 'select',
                'label' => $this->l('Update to Refund Status'),
                'name' => 'paytr_refund',
                'options' => array(
                    'query' => $status_options,
                    'id' => 'id_option',
                    'name' => 'name'
                ),
                'desc' => $this->l('When selected as Enabled, the refunded order\'s status will be changed to refund.')
            ),
            array(
                'type' => 'select',
                'label' => $this->l('Installment Difference Note'),
                'name' => 'paytr_invoice',
                'options' => array(
                    'query' => $status_options,
                    'id' => 'id_option',
                    'name' => 'name'
                ),
                'desc' => $this->l('Installment difference adding to invoice as a note.')
            ),
            array(
                'type' => 'select',
                'label' => $this->l('Installment Options'),
                'name' => 'paytr_installment',
                'options' => array(
                    'query' => $installment_options,
                    'id' => 'id_option',
                    'name' => 'name'
                )
            )
        );
        $category_inputs = array();
        $inputs = array();

        // Category based inputs
        if (Configuration::get('PAYTR_INSTALLMENT') == 13) {
            $finish = array();
            $tree = $this->categoryParser();
            $this->categoryParserClear($tree, 0, array(), $finish);

            foreach ($finish as $key => $val) {
                $category_inputs[] = array(
                    'type' => 'select',
                    'label' => $val,
                    'name' => 'CI[' . $key . ']',
                    'options' => array(
                        'query' => $this->installmentOptions(),
                        'id' => 'id_option',
                        'name' => 'name'
                    )
                );
            }

            $inputs = array_merge($form_inputs, $category_inputs);
        } else {
            $inputs = $form_inputs;
        }

        // Store Information Form
        $fields_form = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('SETTINGS'),
                    'icon' => 'icon-pencil'
                ),
                'input' => $inputs,
                'submit' => array(
                    'title' => $this->l('Save'),
                )
            )
        );

        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitPaytrModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form));
    }

    protected function postValidation()
    {
        if (Tools::isSubmit('submitPaytrModule')) {
            $merchant_id = $this->sanitizeInput(Tools::getValue('paytr_merchant_id'));
            $merchant_key = $this->sanitizeInput(Tools::getValue('paytr_merchant_key'));
            $merchant_salt = $this->sanitizeInput(Tools::getValue('paytr_merchant_salt'));

            // Check Merchant ID != null
            if ($merchant_id == null) {
                $this->postErrors[] = $this->l('Merchant No is required');
            } else if (!is_numeric($merchant_id)) {
                $this->postErrors[] = $this->l('Merchant No must be numeric');
            }
            // Check Merchant Key != null
            if ($merchant_key == null) {
                $this->postErrors[] = $this->l('Merchant Key required');
            } else if (Tools::strlen($merchant_key) != 16) {
                $this->postErrors[] = $this->l('Merchant Key must be 16 char');
            }
            // Check Merchant Salt != null
            if ($merchant_salt == null) {
                $this->postErrors[] = $this->l('Merchant Salt is required');
            } else if (Tools::strlen($merchant_salt) != 16) {
                $this->postErrors[] = $this->l('Merchant Salt must be 16 char');
            }
        }
    }

    protected function postProcess()
    {
        if (Tools::isSubmit('submitPaytrModule')) {
            Configuration::updateValue('PAYTR_MERCHANT_ID', $this->sanitizeInput(Tools::getValue('paytr_merchant_id')));
            Configuration::updateValue('PAYTR_MERCHANT_KEY', $this->sanitizeInput(Tools::getValue('paytr_merchant_key')));
            Configuration::updateValue('PAYTR_MERCHANT_SALT', $this->sanitizeInput(Tools::getValue('paytr_merchant_salt')));
            Configuration::updateValue('PAYTR_LOGO', Tools::getValue('paytr_logo'));

            $languages = Language::getLanguages(false);
            foreach ($languages as $language) {
                Configuration::updateValue('PAYTR_METHOD_TITLE_' . $language['id_lang'], Tools::getValue('paytr_method_title_' . $language['id_lang']));
            }

            Configuration::updateValue('PAYTR_COMPLETED', Tools::getValue('paytr_completed'));
            Configuration::updateValue('PAYTR_CANCELED', Tools::getValue('paytr_canceled'));
            Configuration::updateValue('PAYTR_INVOICE', Tools::getValue('paytr_invoice'));
            Configuration::updateValue('PAYTR_INSTALLMENT', Tools::getValue('paytr_installment'));
            Configuration::updateValue('PAYTR_REFUND', Tools::getValue('paytr_refund'));

            if (Tools::getValue('paytr_installment') == 13) {
                Configuration::updateValue('PAYTR_CI', serialize(Tools::getValue('CI')));
            }

            $this->contentHtml .= $this->displayConfirmation($this->l('Settings updated'));
        }
    }

    protected function sanitizeInput($key)
    {
        return Tools::safeOutput(str_replace(' ', '', preg_replace('/^\p{Z}+|\p{Z}+$/u', ' ', $key)), false);
    }

    public function getContent()
    {
        if (Tools::isSubmit('submitPaytrModule')) {
            $this->postValidation();

            if (!count($this->postErrors)) {
                $this->postProcess();
            } else {
                foreach ($this->postErrors as $err) {
                    $this->contentHtml .= $this->displayError($err);
                }
            }
        }

        $this->contentHtml .= $this->displayFormInfo();
        $this->contentHtml .= $this->displayForm();

        return $this->contentHtml;
    }

    # Category Based Options
    public function categoryParser($start_id = 2)
    {
        $cats = Db::getInstance()->ExecuteS('SELECT c.id_category AS "id", c.id_parent AS "parent_id", cl.name AS "name" FROM ' . _DB_PREFIX_ . 'category c LEFT JOIN ' . _DB_PREFIX_ . 'category_lang cl ON (c.id_category = cl.id_category) ORDER BY cl.name ASC');
        $cat_tree = array();
        foreach ($cats as $key => $item) {
            if ($item['parent_id'] == $start_id) {
                $cat_tree[$item['id']] = array('id' => $item['id'], 'name' => $item['name']);
                $this->parentCategoryParser($cats, $cat_tree[$item['id']]);
            }
        }
        return $cat_tree;
    }

    public function parentCategoryParser(&$cats = array(), &$cat_tree = array())
    {
        foreach ($cats as $key => $item) {
            if ($item['parent_id'] == $cat_tree['id']) {
                $cat_tree['parent'][$item['id']] = array('id' => $item['id'], 'name' => $item['name']);
                $this->parentCategoryParser($cats, $cat_tree['parent'][$item['id']]);
            }
        }
    }

    public function categoryParserClear($tree, $level = 0, $arr = array(), &$finish_him = array())
    {
        foreach ($tree as $id => $item) {
            if ($level == 0) {
                unset($arr);
                $arr = array();
                $arr[] = $item['name'];
            } elseif ($level == 1 or $level == 2) {
                if (count($arr) == ($level + 1)) {
                    $deleted = array_pop($arr);
                }
                $arr[] = $item['name'];
            }
            if ($level < 3) {
                $nav = null;
                foreach ($arr as $key => $val) {
                    $nav .= $val . ($level != 0 ? ' > ' : null);
                }
                $finish_him[$item['id']] = rtrim($nav, ' > ') . '<br>';
                if (!empty($item['parent'])) {
                    $this->categoryParserClear($item['parent'], $level + 1, $arr, $finish_him);
                }
            }
        }
    }

    # Hooks
    public function hookHeader()
    {
        if (Configuration::get('PAYTR_LOGO')) {
            $this->context->controller->addCSS($this->_path . 'views/css/paytr.css');
        }
    }

    public function hookPayment($params)
    {
        $currency_id = $params['cart']->id_currency;
        $currency = new Currency((int)$currency_id);

        if (in_array($currency->iso_code, $this->limited_currencies) == false) {
            return false;
        }

        $this->smarty->assign('ps', array(
            'module_dir' => $this->_path,
            'method_title' => $this->getActionText()
        ));

        return $this->display(__FILE__, 'views/templates/hook/payment.tpl');
    }

    public function hookPaymentReturn($params)
    {
        if (!$this->active) {
            return false;
        }

        if (version_compare(_PS_VERSION_, '1.7.0.0', '<')) {
            $order = $params['objOrder'];

            if ($order->getCurrentOrderState()->id != Configuration::get('PAYTR_CANCELED')) {
                $this->smarty->assign(array(
                    'id_order' => $order->id,
                    'reference' => $order->reference,
                    'params' => $params,
                    'total' => Tools::displayPrice($params['total_to_pay'], $params['currencyObj'], false),
                    'status' => 'ok'
                ));
            } else {
                $this->smarty->assign(array(
                    'status' => 'failed'
                ));
            }

            return $this->display(__FILE__, 'views/templates/hook/confirmation.tpl');
        }
    }

    /**
     * Return payment options available for PS 1.7+
     *
     * @param array Hook parameters
     *
     * @return array|null
     */
    public function hookPaymentOptions($params)
    {
        if (!$this->active) {
            return;
        }

        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $option = new \PrestaShop\PrestaShop\Core\Payment\PaymentOption();
        $option->setCallToActionText($this->getActionText())
            ->setAction($this->context->link->getModuleLink($this->name, 'payment', array(), true));

        if (Configuration::get('PAYTR_LOGO')) {
            $option->setLogo(Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/views/img/logo.png'));
        }

        return [
            $option
        ];
    }

    public function hookDisplayAdminOrder($params)
    {
        $order = new Order($params['id_order']);
        $currency = new Currency($order->id_currency);

        if ($order->module == $this->name) {
            $this->smarty->assign('pt', array(
                'icon_paytr' => Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/views/img/os.gif'),
                'icon_loader' => Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/views/img/loader.gif'),
                'path' => $this->context->link->getModuleLink('paytrcheckout', 'refund'),
                'id_order' => $order->id,
                'currency_icon' => $currency->sign
            ));

            return $this->display(__FILE__, 'views/templates/hook/order.tpl');
        }
    }

    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);
        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }
        return false;
    }

    public function getActionText()
    {
        $lang_id = $this->context->language->id;
        $actionText = Configuration::get('PAYTR_METHOD_TITLE_' . $lang_id);

        if ($actionText == null || $actionText == "") {
            $actionText = 'Kredi / Banka Kartı';
        }

        return $actionText;
    }
}
